/************************************************************************
 * © 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.jdbc.generic;

import static com.sap.cds.DataStoreConfiguration.SUPPORTED_LOCALES;
import static com.sap.cds.impl.localized.LocaleUtils.isLocalized;
import static com.sap.cds.impl.localized.LocaleUtils.localeSpecificViewName;

import java.util.Arrays;
import java.util.Locale;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.DataStoreConfiguration;
import com.sap.cds.impl.localized.LocaleUtils;
import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.jdbc.spi.TableNameResolver;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;

/**
 * A {@link TableNameResolver} that resolves localized strings using
 * locale-specific views
 */
public class LocaleNameTableNameResolver implements TableNameResolver {
	private static final Logger logger = LoggerFactory.getLogger(LocaleNameTableNameResolver.class);

	private static final String DEFAULT_SUPPORTED_LOCALES = "de,fr";
	private final Function<CdsStructuredType, SqlMapping> mapping;
	protected final DataStoreConfiguration dataStoreConfiguration;
	protected final Locale locale;

	public LocaleNameTableNameResolver(Function<CdsStructuredType, SqlMapping> mapping,
			DataStoreConfiguration dataStoreConfiguration, Locale locale) {
		this.mapping = mapping;
		this.dataStoreConfiguration = dataStoreConfiguration;
		this.locale = locale;
	}

	@Override
	public String tableName(CdsEntity entity) {
		SqlMapping sqlMapping = mapping.apply(entity);
		String entityName = entity.getQualifiedName();
		if (locale == null || !isLocalized(entity)) {
			String tableName = sqlMapping.tableName();
			logger.debug("No locale set. Resolve {} to plain table {}.", entityName, tableName);

			return tableName;
		}

		if (hasLocaleSpecificView(locale)) {
			String localeSpecifcViewName = localeSpecificViewName(entityName, locale);
			logger.debug("Locale is {}. Resolve {} using locale-specific view {}.", locale.toLanguageTag(), entityName,
					localeSpecifcViewName);
			return sqlMapping.plainTableName(localeSpecifcViewName);
		}

		String localizedViewName = sqlMapping.localizedViewName();
		logger.debug(
				"Locale is {}. There is no corresponding locale-specific view. Resolve {} using generic localized view {}.",
				locale.toLanguageTag(), entityName, localizedViewName);
		return localizedViewName;
	}

	private boolean hasLocaleSpecificView(Locale locale) {
		if (locale == null) {
			return false;
		}

		String supportedLocales = dataStoreConfiguration.getProperty(SUPPORTED_LOCALES, DEFAULT_SUPPORTED_LOCALES);

		return Arrays.asList(supportedLocales.split(",")).contains(LocaleUtils.getLocaleString(locale));
	}
}
