/*******************************************************************
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.jdbc.hana.search;

import static com.sap.cds.DataStoreConfiguration.SEARCH_MODE_LOCALIZED_ASSOC;
import static com.sap.cds.DataStoreConfiguration.SEARCH_MODE_LOCALIZED_VIEW;
import static com.sap.cds.reflect.impl.reader.model.CdsConstants.ANNOTATION_CDS_SEARCH_MODE;

import java.math.BigDecimal;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import com.sap.cds.DataStoreConfiguration;
import com.sap.cds.impl.builder.model.ListValue;
import com.sap.cds.impl.parser.builder.ExpressionBuilder;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnComparisonPredicate;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.CdsType;

public class HanaSearchResolverUsingScore extends HanaSearchResolver {

	private static final String FUZZINESS_THRESHOLD_ANNOTATION = "@Search.fuzzinessThreshold";
	private static final String EXACT = "EXACT";
	private static final String FUZZY_MIN_SCORE = "FUZZY MINIMAL SCORE";

	// SIMILARITY CALCULATION MODE
	// https://help.sap.com/docs/SAP_HANA_PLATFORM/691cb949c1034198800afde3e5be6570/f3d72af569ab4360b347c73ed4806067.html
	private static final String SCM_SEARCH = "SIMILARITY CALCULATION MODE 'search'";
	private static final String SCM_TYPE_AHEAD = "SIMILARITY CALCULATION MODE 'type ahead'";

	public HanaSearchResolverUsingScore(DataStoreConfiguration config, CdsModel cdsModel, Locale locale) {
		super(config, cdsModel, locale);
	}

	@Override
	protected CqnPredicate searchToHana(Map<CqnElementRef, CdsElement> scoreRefs, CqnPredicate expression) {
		boolean fuzzy = fuzzySearch();
		SearchString searchString = toSearchString(expression, fuzzy);

		ExpressionBuilder scoreBuilder = ExpressionBuilder.create()
				.plain("SCORE(").val(searchString.searchString())
				.plain("IN");

		if (fuzzy && !searchString.containsWildcards()) {
			BigDecimal minScore = fuzzinessThreshold();
			addScoreElements(scoreBuilder, scoreRefs, minScore);
		} else { // EXACT SEARCH
			scoreBuilder.add(ListValue.of(scoreRefs.keySet()));
			scoreBuilder.plain(EXACT);
		}
		scoreBuilder.plain(")");

		return CQL.comparison(scoreBuilder.value(), CqnComparisonPredicate.Operator.GT, CQL.constant(0));
	}

	private static void addScoreElements(ExpressionBuilder score, Map<CqnElementRef, CdsElement> scoreRefs,
			BigDecimal minScore) {
		score.plain("(");

		int i = 0;
		int size = scoreRefs.size();
		for (Entry<CqnElementRef, CdsElement> entry : scoreRefs.entrySet()) {
			CdsElement element = entry.getValue();
			score.add(entry.getKey());
			scoringType(element, score, minScore);
			// TODO support ranking: https://github.tools.sap/cap/dev/issues/1016
			if (++i < size) {
				score.plain(",");
			}
		}
		score.plain(")");
	}

   	private static void scoringType(CdsElement element, ExpressionBuilder score, BigDecimal minScore) {
		minScore = minScore(element, minScore);
		if (minScore.compareTo(BigDecimal.ONE) < 0) {
			score.plain(FUZZY_MIN_SCORE);
			score.constant(minScore);
			// TODO support different SCMs: 'search', 'type ahead', 'search compare'
			// https://help.sap.com/docs/SAP_HANA_PLATFORM/691cb949c1034198800afde3e5be6570/f3d72af569ab4360b347c73ed4806067.html
			CdsType type = element.getType();
			if (fuzzySearchMode(type)) {
				score.plain(SCM_SEARCH);
			} else if (type.isSimpleType(CdsBaseType.UUID)) {
				score.plain(SCM_TYPE_AHEAD);
			}
		} else {
			score.plain(EXACT);
		}
	}

	private static BigDecimal minScore(CdsElement element, BigDecimal minScore) {
		var fuzziness = element.getAnnotationValue(FUZZINESS_THRESHOLD_ANNOTATION, (Number) minScore);
		if (fuzziness instanceof BigDecimal d) {
			return d;
		}
		if (fuzziness instanceof Integer i) {
			return BigDecimal.valueOf(i);
		}
		throw new IllegalArgumentException("Invalid fuzziness threshold: " + fuzziness);
	}

	private static boolean fuzzySearchMode(CdsType type) {
		return type.isSimpleType(CdsBaseType.STRING) || type.isSimpleType(CdsBaseType.LARGE_STRING);
	}

	@Override
	protected boolean needsPushToSubquery(CdsElement element) {
		return false;
	}

	@Override
	protected boolean handleLocalizedElement(CdsStructuredType targetType, Set<CqnElementRef> like,
			Map<CqnElementRef, CdsElement> score, boolean languageGiven, CqnElementRef ref, CdsElement element) {
		if ((languageGiven && isReachableViaLocalizedAssoc(element)) && !searchLocalizedView(targetType)) {
			score.put(ref, element);
			score.put(localizedRef(ref), element);
			return true; // push to subquery
		} else {
			// search localized view
			score.put(ref, element);
		}
		return false;
	}

	private boolean searchLocalizedView(CdsStructuredType targetType) {
		String searchMode = targetType.getAnnotationValue(ANNOTATION_CDS_SEARCH_MODE, configuredSearchMode());

		return SEARCH_MODE_LOCALIZED_VIEW.equalsIgnoreCase(searchMode);
	}

	@Override
	protected void handleRegularElement(CdsStructuredType targetType, Set<CqnElementRef> like,
			Map<CqnElementRef, CdsElement> score, CqnElementRef ref, CdsElement element) {
		score.put(ref, element);
	}

	@Override
	protected void handleLargeStringElement(Set<CqnElementRef> like, Map<CqnElementRef, CdsElement> score,
			CdsStructuredType targetType, CqnElementRef ref, CdsElement element) {
		// SCORE also works for LOBS on disk since HANA QRC 1/2024
		score.put(ref, element);
	}

	@Override
	protected String defaultSearchMode() {
		return SEARCH_MODE_LOCALIZED_ASSOC;
	}

}
