/*******************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.impl;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.DataStoreConfiguration;
import com.sap.cds.SessionContext;
import com.sap.cds.jdbc.spi.DbContext;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.impl.CdsVersion;
import com.sap.cds.util.CdsModelUtils;

public class ContextImpl implements Context {

	private final CdsModel cdsModel;
	private final DbContext dbContext;
	private final CdsVersion compilerVersion;
	private final RuntimeCompatibilityMode runtimeMode;
	private final DataStoreConfiguration dataStoreConfiguration;
	private final SessionContext sessionContext = SessionContext.create();

	@Override
	public void setSessionContext(SessionContext sessionContext) {
		this.sessionContext.from(sessionContext);
	}

	private ContextImpl(CdsModel model, DbContext dbContext, SessionContext sessionContext,
			DataStoreConfiguration dataStoreConfiguration) {
		this.cdsModel = model;
		this.dbContext = dbContext;
		this.sessionContext.from(sessionContext);
		this.compilerVersion = CdsModelUtils.compilerVersion(model);
		this.runtimeMode = new RuntimeCompatibilityMode(compilerVersion.major());
		this.dataStoreConfiguration = dataStoreConfiguration;
	}

	@VisibleForTesting
	public static Context context(CdsModel model, DbContext dbContext, SessionContext sessionContext) {
		return context(model, dbContext, sessionContext, new SystemPropertyDataStoreConfiguration());
	}

	public static Context context(CdsModel model, DbContext dbContext, SessionContext sessionContext,
			DataStoreConfiguration dataStoreConfiguration) {
		return new ContextImpl(model, dbContext, sessionContext, dataStoreConfiguration);
	}

	public static Context context(CdsModel model, DbContext dbContext) {
		return context(model, dbContext, new SystemPropertyDataStoreConfiguration());
	}

	public static Context context(CdsModel model, DbContext dbContext, DataStoreConfiguration dataStoreConfiguration) {
		return new ContextImpl(model, dbContext, SessionContext.create(), dataStoreConfiguration);
	}

	@Override
	public CdsModel getCdsModel() {
		return cdsModel;
	}

	@Override
	public DbContext getDbContext() {
		return dbContext;
	}

	@Override
	public SessionContext getSessionContext() {
		return sessionContext;
	}

	@Override
	public CdsVersion getCdsCompilerVersion() {
		return compilerVersion;
	}

	@Override
	public RuntimeCompatibilityMode getRuntimeMode() {
		return runtimeMode;
	}

	@Override
	public DataStoreConfiguration getDataStoreConfiguration() {
		return dataStoreConfiguration;
	}
}
