/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.impl.sql;

import static com.sap.cds.impl.builder.model.ElementRefImpl.parse;
import static com.sap.cds.impl.localized.LocaleUtils.localizedEntityName;

import java.util.Locale;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.reflect.CdsAnnotation;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;

public class SqlMappingImpl implements SqlMapping {

	public static final String CDS_PERSISTENCE_NAME = "cds.persistence.name";
	private final CdsStructuredType rowType;
	private final Function<String, String> casing;

	public SqlMappingImpl(CdsStructuredType rowType, Function<String, String> casing) {
		this.rowType = rowType;
		this.casing = casing;
	}

	@Override
	public String tableName() {
		CdsEntity entity = asEntity();

		return annotatedName().map(SQLHelper::delimited).orElseGet(() -> plainTableName(entity.getQualifiedName()));
	}

	@Override
	public String localizedViewName() {
		CdsEntity entity = asEntity();

		return annotatedName().filter(n -> !uppercaseOnly(n)).map(n -> SQLHelper.delimited(localizedEntityName(n)))
				.orElseGet(() -> plainTableName(localizedEntityName(entity)));
	}

	private boolean uppercaseOnly(String n) {
		return n.equals(n.toUpperCase(Locale.US));
	}

	private CdsEntity asEntity() {
		if (!(rowType instanceof CdsEntity)) {
			throw new IllegalStateException(rowType.getQualifiedName() + "is no entity");
		}

		return (CdsEntity) rowType;
	}

	private Optional<String> annotatedName() {
		return rowType.<String>findAnnotation(CDS_PERSISTENCE_NAME).map(CdsAnnotation::getValue);
	}

	@Override
	public String plainTableName(String qualifiedEntityName) {
		return delimitedCasing(qualifiedEntityName.replace(".", "_"));
	}

	@Override
	public String columnName(String elementName) {
		return rowType.findElement(elementName).map(this::columnName)
				.orElseGet(() -> delimitedCasing(underscoreSeparated(parse(elementName))));
	}

	@Override
	public String columnName(CqnElementRef ref) {
		return columnName(underscoreSeparated(ref));
	}

	@Override
	public String columnName(Stream<? extends Segment> segments) {
		return columnName(underscoreSeparated(segments));
	}

	@Override
	public String columnName(CdsElement element) {
		Optional<CdsAnnotation<String>> persistenceName = element.findAnnotation(CDS_PERSISTENCE_NAME);
		if (persistenceName.isPresent()) {
			return SQLHelper.delimited(persistenceName.get().getValue());
		}
		return delimitedCasing(element.getName());
	}

	private static String underscoreSeparated(CqnElementRef ref) {
		return underscoreSeparated(ref.stream());
	}

	private static String underscoreSeparated(Stream<? extends Segment> segments) {
		return segments.map(Segment::id).collect(Collectors.joining("_"));
	}

	@Override
	public CdsStructuredType getRowType() {
		return rowType;
	}

	@Override
	public String jsonObjectPath(CqnElementRef ref) {
		return ref.stream().map(seg -> "\"" + seg.id() + "\"").collect(Collectors.joining("."));
	}

	@Override
	public String delimitedCasing(String undelimited) {
		return SQLHelper.delimited(casing.apply(undelimited));
	}

	@Override
	public String columnLikeAlias(CdsElement element) {
		Optional<CdsAnnotation<String>> persistenceName = element.findAnnotation(CDS_PERSISTENCE_NAME);
		return persistenceName.map(stringCdsAnnotation -> SQLHelper.delimited(stringCdsAnnotation.getValue()))
			.orElseGet(() -> casing.apply(element.getName()));
	}

	@Override
	public String cteName() {
		return plainTableName(rowType.getQualifiedName().concat(".CTE"));
	}

}
