/************************************************************************
 * © 2021-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.jdbc.generic;

import static com.sap.cds.util.CdsTypeUtils.dateTime;
import static com.sap.cds.util.CdsTypeUtils.timestamp;

import java.io.InputStream;
import java.io.Reader;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZonedDateTime;
import java.util.Calendar;
import java.util.TimeZone;

import com.sap.cds.jdbc.spi.ValueBinder;
import com.sap.cds.reflect.CdsBaseType;

public abstract class AbstractValueBinder implements ValueBinder {

	protected AbstractValueBinder(int timestampFractionalSeconds, TimeZone timeZone) {
		this.timestampFractionalSeconds = timestampFractionalSeconds;
		this.localCalendar = ThreadLocal.withInitial(() -> Calendar.getInstance(timeZone));
	}

	protected static final ThreadLocal<Calendar> UTC = // NOSONAR
			ThreadLocal.withInitial(() -> Calendar.getInstance(TimeZone.getTimeZone("UTC")));

	private final int timestampFractionalSeconds;
	private final ThreadLocal<Calendar> localCalendar;

	@Override
	@SuppressWarnings("unchecked")
	public <T> Getter<T> getter(CdsBaseType cdsType, boolean isMediaType) {
		if (cdsType != null) {
			return (Getter<T>) getValueFromResult(cdsType, isMediaType);
		}
		return (result, i) -> (T) result.getObject(i);
	}

	@Override
	@SuppressWarnings("unchecked")
	public <T> T getValue(ResultSet result, int i, CdsBaseType cdsType, boolean isMediaType) throws SQLException {
		return (T) getter(cdsType, isMediaType).get(result, i);
	}

	@Override
	public void setValue(PreparedStatement pstmt, int i, CdsBaseType cdsType, Object value) throws SQLException {
		setter(cdsType).set(pstmt, i, value);
	}

	@Override
	public Setter setter(CdsBaseType cdsType) {
		if (cdsType == null) {
			return this::setValue;
		}
		switch (cdsType) {
			case DATE:
				return this::setLocalDate;
			case DATETIME:
				return (pstmt, i, val) -> setInstant(pstmt, i, dateTime(val));
			case TIME:
				return this::setLocalTime;
			case TIMESTAMP:
				return (pstmt, i, val) -> setInstant(pstmt, i,
						timestamp(val, timestampFractionalSeconds));
			case LARGE_BINARY:
				return this::setLargeBinary;
			case HANA_CLOB:
			case LARGE_STRING:
				return this::setLargeString;
			default:
				return PreparedStatement::setObject;
		}
	}

	protected void setValue(PreparedStatement pstmt, int i, Object value) throws SQLException {
		if (value instanceof Instant) {
			setInstant(pstmt, i, (Instant) value);
		} else if (value instanceof LocalDate) {
			setLocalDate(pstmt, i, (LocalDate) value);
		} else if (value instanceof LocalTime) {
			setLocalTime(pstmt, i, (LocalTime) value);
		} else if (value instanceof ZonedDateTime) {
			setZonedDateTime(pstmt, i, (ZonedDateTime) value);
		} else if (value instanceof Timestamp) {
			setInstant(pstmt, i, ((Timestamp) value).toInstant());
		} else if (value instanceof java.sql.Time) {
			setLocalTime(pstmt, i, ((java.sql.Time) value).toLocalTime());
		} else if (value instanceof java.sql.Date) {
			setLocalDate(pstmt, i, ((java.sql.Date) value).toLocalDate());
		} else if (value instanceof byte[]) {
			pstmt.setBytes(i, (byte[]) value);
		} else if (value instanceof Reader) {
			setLargeString(pstmt, i, (Reader) value);
		} else if (value instanceof InputStream) {
			setLargeBinary(pstmt, i, (InputStream) value);
		} else {
			pstmt.setObject(i, value);
		}
	}

	protected void setLocalTime(PreparedStatement pstmt, int i, Object localTime) throws SQLException {
		if (localTime instanceof LocalTime) {
			setLocalTime(pstmt, i, (LocalTime) localTime);
		} else if (localTime instanceof String) {
			setLocalTime(pstmt, i, LocalTime.parse((String) localTime));
		} else if (localTime instanceof java.sql.Time) {
			setLocalTime(pstmt, i, ((java.sql.Time) localTime).toLocalTime());
		} else {
			pstmt.setObject(i, localTime);
		}
	}

	protected abstract void setLocalTime(PreparedStatement pstmt, int i, LocalTime localTime) throws SQLException;

	protected void setLocalDate(PreparedStatement pstmt, int i, Object localDate) throws SQLException {
		if (localDate instanceof LocalDate) {
			setLocalDate(pstmt, i, (LocalDate) localDate);
		} else if (localDate instanceof String) {
			setLocalDate(pstmt, i, LocalDate.parse((String) localDate));
		} else if (localDate instanceof java.sql.Date) {
			setLocalDate(pstmt, i, ((java.sql.Date) localDate).toLocalDate());
		} else {
			pstmt.setObject(i, localDate);
		}
	}

	protected abstract void setLocalDate(PreparedStatement pstmt, int i, LocalDate localDate) throws SQLException;

	protected abstract void setInstant(PreparedStatement pstmt, int i, Instant instant) throws SQLException;

	private void setLargeBinary(PreparedStatement pstmt, int i, Object largeBin) throws SQLException {
		if (largeBin instanceof InputStream) {
			setLargeBinary(pstmt, i, (InputStream) largeBin);
		} else {
			pstmt.setObject(i, largeBin);
		}
	}

	protected abstract void setLargeBinary(PreparedStatement result, int i, InputStream stream) throws SQLException;

	private void setZonedDateTime(PreparedStatement pstmt, int i, ZonedDateTime zonedDateTime) throws SQLException {
		setInstant(pstmt, i, dateTime(zonedDateTime.toInstant()));
	}

	protected abstract LocalTime getLocalTime(ResultSet result, int i) throws SQLException;

	protected abstract LocalDate getLocalDate(ResultSet result, int i) throws SQLException;

	protected abstract Instant getInstant(ResultSet result, int i) throws SQLException;

	private void setLargeString(PreparedStatement pstmt, int i, Object largeStr) throws SQLException {
		if (largeStr instanceof Reader) {
			setLargeString(pstmt, i, (Reader) largeStr);
		} else {
			setString(pstmt, i, largeStr);
		}
	}

	protected abstract void setLargeString(PreparedStatement result, int i, Reader reader) throws SQLException;

	protected abstract Reader getLargeString(ResultSet result, int i) throws SQLException;

	protected abstract InputStream getLargeBinary(ResultSet result, int i) throws SQLException;

	protected Boolean getBoolean(ResultSet result, int i) throws SQLException {
		boolean bool = result.getBoolean(i);
		return result.wasNull() ? null : bool;
	}

	protected Double getDouble(ResultSet result, int i) throws SQLException {
		double dbl = result.getDouble(i);
		return result.wasNull() ? null : dbl;
	}

	protected Short getShort(ResultSet result, int i) throws SQLException {
		short int16 = result.getShort(i);
		return result.wasNull() ? null : int16;
	}

	protected Integer getInt(ResultSet result, int i) throws SQLException {
		int int32 = result.getInt(i);
		return result.wasNull() ? null : int32;
	}

	protected Long getLong(ResultSet result, int i) throws SQLException {
		long int64 = result.getLong(i);
		return result.wasNull() ? null : int64;
	}

	private void setString(PreparedStatement pstmt, int i, Object string) throws SQLException {
		pstmt.setString(i, (String) string);
	}

	protected Float getFloat(ResultSet result, int i) throws SQLException {
		float real = result.getFloat(i);
		return result.wasNull() ? null : real;
	}

	@SuppressWarnings("deprecation")
	private Getter<?> getValueFromResult(CdsBaseType cdsType, boolean isMediaType) {
		switch (cdsType) {
			case BOOLEAN:
				return this::getBoolean;
			case DATE:
				return this::getLocalDate;
			case DATETIME:
				return (result, i) -> dateTime(getInstant(result, i));
			case DECIMAL:
			case HANA_SMALLDECIMAL:
			case DECIMAL_FLOAT:
				return ResultSet::getBigDecimal;
			case DOUBLE:
				return this::getDouble;
			case INTEGER:
			case INT32:
				return this::getInt;
			case INTEGER64:
			case INT64:
				return this::getLong;
			case LARGE_BINARY:
				if (isMediaType) {
					return this::getLargeBinary;
				}
			case HANA_BINARY:
			case BINARY:
				return ResultSet::getBytes;
			case HANA_CLOB:
			case LARGE_STRING:
				if (isMediaType) {
					return this::getLargeString;
				}
			case STRING:
				return ResultSet::getString;
			case TIME:
				return this::getLocalTime;
			case TIMESTAMP:
				return (result, i) -> timestamp(getInstant(result, i), timestampFractionalSeconds);
			case UINT8:
			case INT16:
			case HANA_TINYINT:
			case HANA_SMALLINT:
				return this::getShort;
			case HANA_REAL:
				return this::getFloat;
			default:
				return (result, i) -> result.getObject(i);
		}
	}

	protected Calendar getCalendarForDefaultTimeZone() {
		return localCalendar.get();
	}

}
