/************************************************************************
 * © 2020-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.jdbc.generic;

import static com.sap.cds.impl.localized.LocaleUtils.hasLocalizedElements;

import java.util.Collection;
import java.util.HashSet;
import java.util.Locale;

import com.sap.cds.jdbc.spi.SearchResolver;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;

/**
 * A {@link SearchResolver} implementation for all generic SQL databases except
 * HANA. It either uses the localized_ view generated by the CDS compiler
 * directly. In case each localized element can be resolved by a 'localized' the
 * search expression is resolved via this association at runtime.
 */
public class GenericSearchResolver extends AbstractSearchResolver {

	public GenericSearchResolver(CdsModel cdsModel, Locale locale) {
		super(cdsModel, locale);
	}

	public GenericSearchResolver(CdsModel cdsModel) {
		this(cdsModel, null);
	}

	@Override
	protected void resolve(CqnSelect select, CqnPredicate search, CdsStructuredType targetType,
			Collection<CqnElementRef> searchableRefs) {
		Collection<CqnElementRef> badRefs = new HashSet<>();

		boolean languageIsGiven = locale != null;
		// search non-localized view and additionally texts entity via association
		// "localized"
		final boolean resolveLocalizedViaAssociation = languageIsGiven && hasLocalizedElements(targetType, searchableRefs)

				&& allLocalizedElementsAreReachableViaLocalizedAssociation(targetType, searchableRefs, badRefs)
				&& !hasAliasedLocalizedElementsInView(targetType, searchableRefs, badRefs);

		if (!resolveLocalizedViaAssociation || CdsConstants.ANNOTATION_VALUE_SEARCH_MODE_VIEW.equalsIgnoreCase(
				getSearchModeAnnotation(targetType))) {
			resolveUsingLocalizedViewWithLike(select, search, targetType, searchableRefs, false);
		} else {
			resolveUsingLocalizedAssociationWithLike(select, search, targetType, searchableRefs);
		}

	}

}
