/************************************************************************
 * © 2020-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.jdbc.hana;

import static com.sap.cds.DataStoreConfiguration.HANA_COMPATIBILITY_MODE;
import static com.sap.cds.DataStoreConfiguration.HANA_COMPATIBILITY_MODE_HEX;

import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.CdsDataStoreConnector;
import com.sap.cds.CdsDataStoreConnector.Capabilities;
import com.sap.cds.DataStoreConfiguration;
import com.sap.cds.jdbc.generic.GenericDbContext;
import com.sap.cds.jdbc.hana.search.HanaSearchResolver;
import com.sap.cds.jdbc.spi.ExceptionAnalyzer;
import com.sap.cds.jdbc.spi.PredicateMapper;
import com.sap.cds.jdbc.spi.SearchResolver;
import com.sap.cds.jdbc.spi.SessionVariableSetter;
import com.sap.cds.jdbc.spi.StatementResolver;
import com.sap.cds.jdbc.spi.ValueBinder;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.reflect.CdsModel;

public final class HanaDbContext extends GenericDbContext {
	public static final String USE_HEX_PLAN = "hdb.USE_HEX_PLAN";
	public static final String NO_USE_HEX_PLAN = "hdb.NO_USE_HEX_PLAN";
	private static final Logger logger = LoggerFactory.getLogger(HanaDbContext.class);
	private final int majorVersion;
	private boolean optimizeForHexEngine;

	public HanaDbContext(DataStoreConfiguration dataStoreConfiguration, int majorVersion) {
		super(dataStoreConfiguration);
		this.majorVersion = majorVersion;
		this.optimizeForHexEngine = majorVersion >= 4
				&& HANA_COMPATIBILITY_MODE_HEX.equals(config.getProperty(HANA_COMPATIBILITY_MODE));
	}

	@Override
	public SearchResolver createSearchResolver(CdsModel model, Locale locale, CqnSelect select) {
		boolean useLocalizedViewSearchResolver = config
				.getProperty(DataStoreConfiguration.USE_LOCALIZED_VIEW_SEARCH_RESOLVER, false);
		if (useLocalizedViewSearchResolver) {
			logger.debug("Switched to LocalizedViewSearchResolver due to provided configuration.");
			return new LocalizedViewSearchResolver(model, locale);
		}
		Map<String, Object> hints = select.hints();
		if (hints.containsKey(NO_USE_HEX_PLAN)) {
			return HanaSearchResolver.forLegacyEngine(model, locale);
		}
		if (hints.containsKey(USE_HEX_PLAN) || optimizeForHexEngine) {
			return HanaSearchResolver.forHexEngine(model, locale);
		}
		return HanaSearchResolver.forLegacyEngine(model, locale);
	}

	@Override
	public StatementResolver getStatementResolver() {
		return new HanaStatementResolver(config, majorVersion, optimizeForHexEngine);
	}

	@Override
	public PredicateMapper getPredicateMapper() {
		return new HanaPredicateMapper();
	}

	@Override
	public ExceptionAnalyzer getExceptionAnalyzer() {
		return new HanaExceptionAnalyzer();
	}

	@Override
	public Capabilities getCapabilities() {
		return CdsDataStoreConnector.Capabilities.ALL;
	}

	@Override
	public ValueBinder getBinder(TimeZone timeZone) {
		return new HanaBinder(timeZone);
	}

	@Override
	public SessionVariableSetter getSessionVariableSetter() {
		return new HanaSessionVariableSetter();
	}
}
