/************************************************************************
 * © 2020-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.jdbc.sqlite;

import static com.sap.cds.impl.sql.SQLHelper.commaSeparated;
import static java.util.stream.Collectors.joining;

import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.sap.cds.jdbc.spi.StatementResolver;
import com.sap.cds.ql.cqn.CqnLock;

public class SqliteStatementResolver implements StatementResolver {

	private static final Optional<String> EMPTY = Optional.empty();

	@Override
	public Optional<String> collate(Locale locale) {
		if (locale != null) {
			return Optional.of("COLLATE NOCASE");
		}
		return EMPTY;
	}

	/**
	 * UPSERT is a clause added to INSERT that causes the INSERT to behave as an
	 * UPDATE or a no-op if the INSERT would violate a uniqueness constraint. UPSERT
	 * is not standard SQL. UPSERT in SQLite follows the syntax established by
	 * PostgreSQL, with generalizations.
	 * 
	 * https://www.sqlite.org/lang_upsert.html
	 * 
	 * The REPLACE command is not used, since it deletes and inserts:
	 * https://www.sqlite.org/lang_replace.html
	 * https://www.sqlite.org/lang_conflict.html
	 */
	@Override
	public String upsert(String table, Stream<String> keyColumns, Stream<String> upsertColumns,
			Stream<String> upsertValues) {
		List<String> col = upsertColumns.collect(Collectors.toList());
		String columns = commaSeparated(col.stream());
		String insertValues = commaSeparated(upsertValues);
		String conflictTarget = commaSeparated(keyColumns);
		String updateValues = commaSeparated(col.stream().map(c -> "EXCLUDED." + c));

		return Stream.of("INSERT INTO", table, columns, "VALUES", insertValues, "ON CONFLICT", conflictTarget,
				"DO UPDATE SET", columns, "=", updateValues).collect(joining(" "));
	}

	@Override
	public Stream<String> lockClause(CqnLock lock) {
		return Stream.empty();
	}

}
