/*
 * © 2018-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.impl;

import static com.sap.cds.ql.cqn.CqnElementRef.$KEY;
import static com.sap.cds.util.CdsModelUtils.element;

import com.google.common.collect.Sets;
import com.sap.cds.CdsDataStoreConnector.Capabilities;
import com.sap.cds.impl.docstore.DocStoreUtils;
import com.sap.cds.impl.draft.DraftUtils;
import com.sap.cds.ql.CdsDataException;
import com.sap.cds.ql.cqn.CqnDelete;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnInsert;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnUpsert;
import com.sap.cds.ql.cqn.CqnValidationException;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.reflect.impl.DraftAdapter;
import com.sap.cds.reflect.impl.reader.model.CdsConstants;
import com.sap.cds.util.CdsModelUtils;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class CqnValidatorImpl implements CqnValidator {

  private final DocStoreUtils docStoreUtils;
  private final boolean supportsParamViews;

  public CqnValidatorImpl(CdsModel cdsModel, Capabilities capabilities) {
    this.docStoreUtils = new DocStoreUtils(cdsModel);
    this.supportsParamViews = capabilities.supportsViewsWithParameters();
  }

  @Override
  public void validate(CqnSelect query, CdsEntity entity) {
    if (!supportsParamViews && entity != null && query.from().isRef()) {
      if (entity.isView() && entity.params().count() > 0) {
        throw new UnsupportedOperationException(
            "Parametrized views are not supported by this data store");
      }
    }
  }

  @Override
  public void validate(CqnInsert insert, CdsEntity entity) {
    if (docStoreUtils.targetsDocStore(insert)) {
      return; // no need to validate insert data for docstore collections
    }
    validateElements(entity, insert.elements(entity));
  }

  @Override
  public void validate(CqnUpdate update, CdsEntity entity) {
    if (docStoreUtils.targetsDocStore(update)) {
      assertThatAllEntriesHaveSameStructure(update, entity);
      return; // no need to validate update data for docstore collections
    }
    validateElements(entity, update.elements(entity));

    update.where().ifPresent(w -> w.accept(new VirtualElementValidator(entity)));
  }

  @Override
  public void validate(CqnUpsert upsert, CdsEntity entity) {
    Set<String> keyNames = CdsModelUtils.concreteKeyNames(entity);
    if (DraftUtils.isDraftEnabled(entity) && !DraftUtils.isDraftView(entity)) {
      keyNames.remove(DraftAdapter.IS_ACTIVE_ENTITY);
    }
    var upsertElements = upsert.elements(entity).collect(Collectors.toSet());
    if (!upsertElements.containsAll(keyNames)) {
      Set<String> missingKeys = Sets.difference(keyNames, upsertElements);
      var paths = missingKeys.stream().map(k -> k.replace('_', '.')).toList();
      if (!upsertElements.containsAll(paths)) {
        throw new CdsDataException("Missing key value for " + missingKeys);
      }
    }
  }

  private void assertThatAllEntriesHaveSameStructure(CqnUpdate update, CdsEntity entity) {
    Set<String> elements = update.data().keySet();
    if (!update.entries().stream().allMatch(e -> e.keySet().equals(elements))) {
      String allElements = update.elements(entity).collect(Collectors.joining(", ", "[", "]"));
      throw new UnsupportedOperationException(
          "Each bulk update entry must contain values for the same elements " + allElements);
    }
  }

  private void validateElements(CdsEntity entity, Stream<String> usedElements) {
    if (entity.findAnnotation(CdsConstants.OPEN).isEmpty()) {
      usedElements.forEach(
          element -> {
            int firstDot = element.indexOf('.');
            String rootElement = firstDot == -1 ? element : element.substring(0, firstDot);
            if (entity.findElement(element).isEmpty()
                && entity.findElement(rootElement).isEmpty()) {
              throw new CqnValidationException(
                  "Element '"
                      + element
                      + "' does not exist in entity '"
                      + entity.getQualifiedName()
                      + "'");
            }
          });
    }
  }

  @Override
  public void validate(CqnDelete delete, CdsEntity entity) {
    delete.where().ifPresent(w -> w.accept(new VirtualElementValidator(entity)));
  }

  private class VirtualElementValidator implements CqnVisitor {
    private CdsStructuredType root;

    public VirtualElementValidator(CdsStructuredType rowType) {
      root = rowType;
    }

    @Override
    public void visit(CqnElementRef ref) {
      if ($KEY.equals(ref.lastSegment())
          || CdsModelUtils.isEtagPlaceholder(ref)
          || CdsModelUtils.isContextElementRef(ref)) {
        return;
      }
      if (element(root, ref).isVirtual()) {
        throw virtualElementException(ref.displayName());
      }
    }
  }

  public static RuntimeException virtualElementException(String element) {
    return new CqnValidationException(
        "Virtual element '%s' is not allowed in filter conditions".formatted(element));
  }
}
