/*
 * © 2021-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.impl.docstore;

import static com.sap.cds.impl.sql.SQLStatementBuilder.commaSeparated;
import static com.sap.cds.impl.sql.SpaceSeparatedCollector.joining;

import com.sap.cds.impl.Context;
import com.sap.cds.impl.PreparedCqnStmt;
import com.sap.cds.impl.sql.SQLStatementBuilder;
import com.sap.cds.impl.sql.TokenToSQLTransformer;
import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.reflect.CdsEntity;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

public class DocStoreSelectStatementBuilder implements DocStoreStatementBuilder {
  private final CqnSelect select;
  private final TokenToSQLTransformer toSql;
  private final SqlMapping sqlMapping;
  private final String tableName;
  private final CdsEntity entity;
  private final List<PreparedCqnStmt.Parameter> params = new ArrayList<>();

  public DocStoreSelectStatementBuilder(Context context, CqnSelect select) {
    this.entity = context.getCdsModel().getEntity(select.ref().firstSegment());
    this.select = select;
    this.sqlMapping = context.getDbContext().getSqlMapping(entity);
    this.tableName = this.sqlMapping.tableName();
    this.toSql =
        TokenToSQLTransformer.notCollating(
            context,
            (clause, ref) -> Stream.of(sqlMapping.columnName(ref)),
            entity,
            this.tableName,
            params);
  }

  @Override
  public SQLStatementBuilder.SQLStatement build() {

    Stream.Builder<String> builder = Stream.builder();
    builder.add("SELECT * FROM");
    builder.add(tableName);

    select
        .where()
        .map(toSql::toSQL)
        .ifPresent(
            whereSql -> {
              builder.add("WHERE");
              builder.add(whereSql);
            });

    List<CqnSortSpecification> orderBy = select.orderBy();
    if (!orderBy.isEmpty()) {
      builder.add("ORDER BY");
      commaSeparated(orderBy.stream(), this::sort).forEach(builder);
    }

    String sql = builder.build().collect(joining());

    return new SQLStatementBuilder.SQLStatement(sql, params);
  }

  private String sort(CqnSortSpecification o) {
    StringBuilder sort = new StringBuilder(toSql.apply(o.value()));
    switch (o.order()) {
      case DESC:
      case DESC_NULLS_FIRST:
        sort.append(" DESC");
        break;
      case ASC_NULLS_LAST:
      default: // ASC
    }
    return sort.toString();
  }
}
