/*
 * © 2018-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.impl.qat;

import com.sap.cds.CqnTableFunction;
import com.sap.cds.impl.Context;
import com.sap.cds.impl.PreparedCqnStmt.CqnParam;
import com.sap.cds.impl.PreparedCqnStmt.Parameter;
import com.sap.cds.impl.draft.DraftUtils;
import com.sap.cds.impl.draft.DraftUtils.Element;
import com.sap.cds.impl.sql.SQLStatementBuilder.SQLStatement;
import com.sap.cds.impl.sql.SelectStatementBuilder;
import com.sap.cds.impl.sql.TokenToSQLTransformer;
import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.jdbc.spi.TableFunctionMapper;
import com.sap.cds.jdbc.spi.TableNameResolver;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnReference;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnVisitor;
import com.sap.cds.ql.impl.SelectBuilder;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsSimpleType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.util.CdsModelUtils;
import java.util.ArrayDeque;
import java.util.Collections;
import java.util.Deque;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class FromClauseBuilder {
  private static final String INNER = "INNER";
  private static final String LEFT = "LEFT";
  private static final String OUTER = "OUTER";
  private static final String JOIN = "JOIN";
  private static final String ON = "ON";
  private static final String LPAREN = "(";
  private static final String RPAREN = ")";
  private static final String AND = "and";

  private final Context context;
  private final List<Parameter> params;
  private final Map<String, Object> hints;
  private final Map<String, CTE> ctes;

  public FromClauseBuilder(Context context, List<Parameter> params) {
    this(context, params, new HashMap<>(), Collections.emptyMap());
  }

  public FromClauseBuilder(
      Context context, List<Parameter> params, Map<String, CTE> ctes, Map<String, Object> hints) {
    this.context = context;
    this.params = params;
    this.hints = hints;
    this.ctes = ctes;
  }

  public Stream<String> sql(Deque<QatSelectableNode> outer) {
    ToSQLVisitor v = new ToSQLVisitor(outer);
    QatTraverser.take(v).traverse(outer.getLast());

    return v.sql();
  }

  private SqlMapping getSqlMapping(CdsStructuredType targetType) {
    return context.getDbContext().getSqlMapping(targetType);
  }

  private class ToSQLVisitor implements QatVisitor {

    final Deque<QatSelectableNode> outer;
    final Stream.Builder<String> sql = Stream.builder();

    public ToSQLVisitor(Deque<QatSelectableNode> outer) {
      this.outer = outer;
    }

    @Override
    public void visit(QatEntityRootNode root) {
      assertFilterHasNoPaths(root);

      table(root);
    }

    @Override
    public void visit(QatSelectRootNode subquery) {
      subSelect(subquery.select(), subquery.alias());
    }

    @Override
    public void visit(QatTableFunctionRootNode tableFunction) {
      tableFunction(tableFunction.tableFunction(), tableFunction.alias());
    }

    @Override
    public void visit(QatAssociationNode assoc) {
      assertFilterHasNoPaths(assoc);

      if (assoc.inSource()) {
        add(INNER);
      } else {
        if (assoc.skipJoin()) {
          return;
        }
        add(LEFT);
        add(OUTER);
      }
      add(JOIN);
      table(assoc);
      add(ON);
      if (assoc.inSource()) {
        onConditionFilterOnParent(assoc);
      } else {
        onConditionFilterOnTarget(assoc);
      }
    }

    private void assertFilterHasNoPaths(QatEntityNode node) {
      node.filter()
          .ifPresent(
              f ->
                  f.accept(
                      new CqnVisitor() {
                        @Override
                        public void visit(CqnElementRef ref) {
                          if (ref.size() > 1 && !CdsModelUtils.isContextElementRef(ref)) {
                            throw new UnsupportedOperationException(
                                "Path " + ref + " in infix filter is not supported");
                          }
                        }
                      }));
    }

    private void onConditionFilterOnParent(QatAssociationNode assoc) {
      onCondition(assoc, (QatEntityNode) assoc.parent());
    }

    private void onConditionFilterOnTarget(QatAssociationNode assoc) {
      onCondition(assoc, assoc);
    }

    private void onCondition(QatAssociationNode assoc, QatEntityNode filtered) {
      CqnPredicate on = assoc.association().onCondition();
      TokenToSQLTransformer onToSQL =
          TokenToSQLTransformer.notCollating(context, params, refInOnToAlias(assoc), outer);
      Optional<CqnPredicate> filter = filtered.filter();
      if (filter.isPresent()) {
        add(LPAREN);
        add(onToSQL.toSQL(assoc.entity(), on)); // toSQL
        add(RPAREN);
        TokenToSQLTransformer filterToSQL =
            TokenToSQLTransformer.notCollating(
                context, params, refInFilterToAlias(filtered), outer);

        filter
            .map(filterToSQL::toSQL)
            .ifPresent(
                f -> {
                  // not simplified to TRUE
                  add(AND);
                  add(LPAREN);
                  add(f);
                  add(RPAREN);
                });
      } else {
        add(onToSQL.toSQL(on)); // toSQL
      }
    }

    private void add(String txt) {
      sql.add(txt);
    }

    private Ref2Column refInFilterToAlias(QatEntityNode entityNode) {
      // we assume that filter cannot be applied to subqueries
      String alias = entityNode.alias();
      CdsEntity entity = entityNode.rowType();
      SqlMapping toSql = getSqlMapping(entity);

      return (clause, ref) -> {
        checkRef(ref);

        String elementName = ref.lastSegment();
        return Stream.of(alias + "." + toSql.columnName(elementName));
      };
    }

    private Ref2Column refInOnToAlias(QatAssociationNode assoc) {
      String lhs = parentAlias(assoc);
      String rhs = assoc.alias();
      CdsStructuredType lhsType = ((QatStructuredNode) assoc.parent()).rowType();
      CdsEntity rhsType = assoc.rowType();
      SqlMapping lhsToSql = getSqlMapping(lhsType);
      SqlMapping rhsToSql = getSqlMapping(rhsType);

      return (clause, ref) -> {
        checkRef(ref);

        String alias;
        String columnName;
        Stream<? extends CqnReference.Segment> streamOfSegments = ref.stream();
        if (ref.size() > 1) {
          streamOfSegments = streamOfSegments.skip(1);
        }
        if (ref.firstSegment().equals(assoc.name())) {
          alias = rhs;
          columnName = rhsToSql.columnName(streamOfSegments);
        } else {
          alias = lhs;
          columnName = lhsToSql.columnName(streamOfSegments);
        }

        return Stream.of(alias + "." + columnName);
      };
    }

    private String parentAlias(QatAssociationNode assoc) {
      QatNode parent = assoc.parent();
      if (parent instanceof QatSelectableNode sn) {
        return sn.alias();
      } else if (parent instanceof QatStructuredElementNode sen) {
        // TODO #2032 association in structured element
        throw new UnsupportedOperationException(
            "Using association '%s' from structured parent element '%s' is not supported"
                .formatted(assoc.name(), parent.name()));
      }
      throw new IllegalStateException(
          "parent '%s' of association node '%s' has unexpected type %s"
              .formatted(parent.name(), assoc.name(), parent.getClass().getSimpleName()));
    }

    private void subSelect(CqnSelect select, String alias) {
      add(LPAREN);
      SQLStatement stmt =
          SelectStatementBuilder.forSubquery(
                  context, params, select, new ArrayDeque<>(), ctes, false)
              .build();
      add(stmt.sql());
      add(RPAREN);
      add(alias);
    }

    private void tableFunction(CqnTableFunction tableFunction, String alias) {
      TableFunctionMapper mapper =
          context.getDbContext().getTableFunctionMapper(context, params, tableFunction);
      String fromSQL = mapper.toSQL();
      add(fromSQL);
      add(alias);
    }

    private String viewName(
        CdsEntity cdsEntity,
        EnumSet<DraftUtils.Element> draftElements,
        boolean usesLocalizedElements) {
      String tableName = tableName(cdsEntity, draftElements, usesLocalizedElements);
      String localParams =
          cdsEntity
              .params()
              .map(
                  p -> {
                    CqnParam param = new CqnParam(p.getName(), p.getDefaultValue().orElse(null));
                    param.type(p.getType().as(CdsSimpleType.class).getType());
                    FromClauseBuilder.this.params.add(param);
                    return "?";
                  })
              .collect(Collectors.joining(","));
      if (!localParams.isEmpty()) {
        tableName += "(" + localParams + ")";
      }

      return tableName;
    }

    private String tableName(
        CdsEntity cdsEntity,
        EnumSet<DraftUtils.Element> draftElements,
        boolean usesLocalizedElements) {
      TableNameResolver resolver;
      // TODO: actually, we should use the table name resolver from the DbContext and
      // not switch this
      // at runtime. This is, however, needed for the SearchResolver logic. For most
      // of the cases its
      // ok to skip localized views and just use the plain table name. At the moment
      // this runtime dynamic
      // aspect is achieved with a not completely ready DB execution hints logic. This
      // needs some concept work.
      if (!usesLocalizedElements || ignoreLocalizedViews()) {
        resolver = new PlainTableNameResolver();
      } else {
        resolver = context.getTableNameResolver();
      }

      if (!ignoreDraftSubquery()
          && DraftUtils.isDraftEnabled(cdsEntity)
          && !DraftUtils.isDraftView(cdsEntity)) {
        // TODO: consider search without localized views?
        return DraftUtils.activeEntity(context, resolver, cdsEntity, draftElements);
      }
      return resolver.tableName(cdsEntity);
    }

    private boolean ignoreLocalizedViews() {
      return (Boolean) hints.getOrDefault(SelectBuilder.IGNORE_LOCALIZED_VIEWS, false);
    }

    private boolean ignoreDraftSubquery() {
      return (Boolean) hints.getOrDefault(SelectBuilder.IGNORE_DRAFT_SUBQUERIES, false);
    }

    private void table(QatEntityNode node) {
      // CTE replacement is indexed. Docstore is evaluated first
      if (!ctes.containsKey(node.alias())) {
        // If the entity is replaced by CTE, use it for every node
        CdsEntity entity = node.rowType();
        if (ctes.containsKey(entity.getQualifiedName())) {
          String cteName = ctes.get(entity.getQualifiedName()).name();
          if (!ignoreDraftSubquery() && DraftUtils.isActive(entity)) {
            cteName = DraftUtils.activeEntity(context, cteName, entity, draftElements(node));
          }
          add(cteName);
        } else {
          String viewName = viewName(entity, draftElements(node), usesLocalizedElements(node));
          add(viewName);
        }
      }
      add(node.alias()); // Covered by CTE for DocStore
    }

    private Stream<String> sql() {
      return sql.build();
    }
  }

  private static EnumSet<DraftUtils.Element> draftElements(QatEntityNode node) {
    EnumSet<Element> draftElements = EnumSet.noneOf(DraftUtils.Element.class);
    for (QatNode child : node.children()) {
      child.accept(
          new QatVisitor() {
            @Override
            public void visit(QatElementNode elementNode) {
              DraftUtils.draftElement(elementNode.element()).ifPresent(draftElements::add);
            }

            @Override
            public void visit(QatAssociationNode association) {
              QatAssociation assoc = association.association();
              assoc
                  .on()
                  .ifPresent(
                      o ->
                          o.accept(
                              new CqnVisitor() {
                                @Override
                                public void visit(CqnElementRef ref) {
                                  if (ref.size() == 1) {
                                    DraftUtils.draftElement(ref.lastSegment())
                                        .ifPresent(draftElements::add);
                                  }
                                }
                              }));
            }
          });
    }

    return draftElements;
  }

  private static boolean usesLocalizedElements(QatEntityNode node) {
    CdsEntity entity = node.rowType();
    boolean[] usesLocalized = new boolean[1];

    CqnVisitor checkRefs =
        new CqnVisitor() {
          @Override
          public void visit(CqnElementRef ref) {
            if (ref.size() == 1
                && CdsModelUtils.findElement(entity, ref)
                    .map(CdsElement::isLocalized)
                    .orElse(false)) {
              usesLocalized[0] = true;
            }
          }
        };

    QatVisitor vElement =
        new QatVisitor() {
          @Override
          public void visit(QatStructuredElementNode struct) {
            struct.children().forEach(c -> c.accept(this));
          }

          @Override
          public void visit(QatElementNode element) {
            if (element.element().isLocalized()) {
              usesLocalized[0] = true;
            }
          }

          @Override
          public void visit(QatAssociationNode assoc) {
            // check for localized elements on source side
            assoc.association().on().ifPresent(o -> o.accept(checkRefs));
          }
        };

    QatVisitor vEntity =
        new QatVisitor() {
          @Override
          public void visit(QatEntityRootNode root) {
            root.filter().ifPresent(f -> f.accept(checkRefs));
            checkChildNodes(root);
          }

          @Override
          public void visit(QatAssociationNode assoc) {
            assoc.filter().ifPresent(f -> f.accept(checkRefs));
            checkChildNodes(assoc);

            // check for localized elements on target side
            CqnVisitor collectRefs =
                new CqnVisitor() {
                  @Override
                  public void visit(CqnElementRef ref) {
                    if (ref.size() == 2
                        && ref.firstSegment().equals(assoc.name())
                        && assoc
                            .rowType()
                            .findElement(ref.lastSegment())
                            .map(CdsElement::isLocalized)
                            .orElse(false)) {
                      usesLocalized[0] = true;
                    }
                  }
                };
            assoc.association().on().ifPresent(o -> o.accept(collectRefs));
          }

          private void checkChildNodes(QatEntityNode node) {
            Iterator<QatNode> iter = node.children().iterator();
            while (!usesLocalized[0] && iter.hasNext()) {
              iter.next().accept(vElement);
            }
          }
        };

    node.accept(vEntity);

    return usesLocalized[0];
  }

  private static void checkRef(CqnElementRef ref) {
    if (CdsModelUtils.isContextElementRef(ref)) {
      throw new IllegalStateException("Can't convert context element ref to column name");
    }
  }

  /**
   * A table name resolver that the plain {@link SqlMapping} table name for the given entity. This
   * is only used if a hint has been set that requires plain rendering without e.g. localized views.
   */
  private class PlainTableNameResolver implements TableNameResolver {

    @Override
    public String tableName(CdsEntity entity) {
      return getSqlMapping(entity).tableName();
    }
  }
}
