/*
 * © 2018-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.impl.qat;

import com.google.common.base.Strings;
import com.sap.cds.impl.builder.model.ElementRefImpl;
import com.sap.cds.impl.docstore.DocStoreUtils;
import com.sap.cds.impl.localized.LocaleUtils;
import com.sap.cds.impl.sql.SelectStatementBuilder;
import com.sap.cds.impl.sql.SqlMappingImpl;
import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnValidationException;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.OnConditionAnalyzer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class Ref2QualifiedColumn implements Ref2Column {

  private final QatSelectableNode root;
  private final QatSelectableNode outer;
  private final String rootName;
  private final SqlMapping sqlMapping;
  private final LocaleUtils localeUtils;
  private String collateClause = null;

  public Ref2QualifiedColumn(
      Function<CdsStructuredType, SqlMapping> mapping,
      Deque<QatSelectableNode> outer,
      LocaleUtils localeUtils) {
    Iterator<QatSelectableNode> reverse = outer.descendingIterator();
    this.root = qatRoot(reverse.next());
    this.rootName = root.rowType().getQualifiedName();
    this.outer = reverse.hasNext() ? reverse.next() : null;
    this.sqlMapping = mapping.apply(root.rowType());
    this.localeUtils = localeUtils;
  }

  public void startCollate(String collateClause) {
    if (Strings.emptyToNull(collateClause) != null) {
      this.collateClause = " " + collateClause;
    } else {
      stopCollate();
    }
  }

  public void stopCollate() {
    this.collateClause = null;
  }

  public Stream<String> apply(CqnElementRef ref) {
    return apply(Clause.WHERE, ref);
  }

  @Override
  public Stream<String> apply(Clause clause, CqnElementRef ref) {
    List<? extends Segment> segments = ref.segments();
    CdsBaseType refType = ref.type().map(CdsBaseType::cdsType).orElse(null);
    if (ref.firstSegment().equals(CqnExistsSubquery.OUTER)) {
      return qualifiedColumnName(clause, segments.subList(1, segments.size()), outer, refType, ref);
    }

    return qualifiedColumnName(clause, segments, root, refType, ref);
  }

  private Stream<String> qualifiedColumnName(
      Clause clause,
      List<? extends Segment> segments,
      QatNode node,
      CdsBaseType type,
      CqnElementRef ref) {
    boolean firstSegment = segments.size() > 1;
    List<String> columnNames = new ArrayList<>(2);
    String jsonPath = null;

    for (int i = 0; i < segments.size(); i++) {
      Segment s = segments.get(i);
      if (firstSegment && s.id().equals(rootName)) {
        continue;
      }
      if (node instanceof QatElementNode en && en.isJson()) {
        jsonPath =
            segments.stream().skip(i).map(Segment::id).collect(Collectors.joining(".", "$.", ""));
        break;
      }

      node = node.child(s.id(), s.filter());
      assertNotNull(segments, node);
      firstSegment = false;
    }

    if (node instanceof QatElementNode en) {
      if (!en.element().isVirtual()) {
        columnNames.add(columnName(en));
      } else if (clause == Clause.WHERE) {
        throw new CqnValidationException("Virtual element '" + en.element() + "' is not allowed");
      }
    } else if (node instanceof QatAssociationNode an) {
      QatAssociation association = an.association();
      CdsStructuredType parent = root.rowType();
      if (an.parent() instanceof QatStructuredElementNode sn) {
        parent = sn.rowType(); // assoc in structured element
      }
      if (CdsModelUtils.isReverseAssociation(parent.getAssociation(association.name()))) {
        return Stream.empty(); // reverse mapped association
      }
      addFkColumns(ref, an, association, columnNames);
    } else {
      throw new CqnValidationException(node.name() + " does not refer to an element");
    }
    do {
      node = node.parent();
    } while (node != null && !(node instanceof QatSelectableNode));
    assertNotNull(segments, node);

    var tableAlias = ((QatSelectableNode) node).alias();
    var columns = columnNames.stream().map(c -> column(tableAlias, c));
    if (!Strings.isNullOrEmpty(jsonPath)) {
      var jsPath = jsonPath;
      if (type != null && type != CdsBaseType.MAP) {
        columns = columns.map(col -> sqlMapping.jsonValue(col, jsPath, type));
      } else {
        columns =
            columns.map(
                col ->
                    switch (clause) {
                      case WHERE, ORDERBY -> sqlMapping.jsonValue(col, jsPath, type);
                      case SELECT -> sqlMapping.jsonQuery(col, jsPath);
                    });
      }
    }
    return columns;
  }

  private void addFkColumns(
      CqnElementRef ref,
      QatAssociationNode an,
      QatAssociation association,
      List<String> columnNames) {
    new OnConditionAnalyzer(association.name(), association.onCondition(), false)
        .tryGetFkMapping().entrySet().stream()
            .filter(e -> !CdsModelUtils.isContextElementRef(CQL.get(e.getKey())))
            .map(
                e ->
                    column(an, e)
                        + SelectStatementBuilder.ColumnAlias.alias(ref, e.getValue().asRef()))
            .forEach(columnNames::add);
  }

  private String column(QatAssociationNode an, Map.Entry<String, CqnValue> fk) {
    var f = fk.getKey();
    var parent = an.parent();
    while (parent instanceof QatStructuredElementNode) {
      f = parent.name() + "_" + f;
      parent = parent.parent();
    }
    return sqlMapping.columnName(f);
  }

  private static String column(String alias, String col) {
    return alias + "." + col;
  }

  private static QatSelectableNode qatRoot(QatNode root) {
    Optional<QatNode> next;
    while ((next = childInSource(root)).isPresent()) {
      root = next.get();
    }

    return (QatSelectableNode) root;
  }

  private static Optional<QatNode> childInSource(QatNode root) {
    return root.children().stream().filter(QatNode::inSource).findFirst();
  }

  private String columnName(QatElementNode elementNode) {
    CdsElement cdsElement = elementNode.element();
    String columnName = sqlMapping.columnName(cdsElement);
    if (elementNode.parent() instanceof QatStructuredElementNode) {
      if (DocStoreUtils.targetsDocStore((CdsStructuredType) cdsElement.getDeclaringType())) {
        return columnName;
      }
      if (cdsElement.findAnnotation(SqlMappingImpl.CDS_PERSISTENCE_NAME).isEmpty()) {
        columnName = structuredColumnName(elementNode);
      }
    }
    // TODO extract COLLATE from column name resolver (issues/1071)
    if (collateClause != null && localeUtils.requiresCollate(cdsElement)) {
      columnName = columnName + collateClause;
    }

    return columnName;
  }

  private String structuredColumnName(QatNode node) {
    List<String> elementName = new ArrayList<>();
    elementName.add(((QatElementNode) node).element().getName());
    QatNode parent = node.parent();
    while (parent instanceof QatStructuredElementNode) {
      QatStructuredElementNode structNode = (QatStructuredElementNode) parent;
      elementName.add(structNode.element().getName());
      parent = structNode.parent();
    }
    Collections.reverse(elementName);
    return sqlMapping.delimitedCasing(String.join("_", elementName));
  }

  private static void assertNotNull(List<? extends Segment> segments, QatNode node) {
    if (node == null) {
      CqnElementRef ref = ElementRefImpl.elementRef(segments, null, null);
      throw new CqnValidationException("Unresolvable path expression: " + ref.toJson());
    }
  }

  public String rootAlias() {
    return root.alias();
  }
}
