/*
 * © 2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.impl.sql;

import static com.sap.cds.impl.docstore.DocStoreUtils.targetsDocStore;
import static com.sap.cds.ql.impl.SelectListValueBuilder.select;

import com.sap.cds.impl.Context;
import com.sap.cds.impl.PreparedCqnStmt;
import com.sap.cds.impl.docstore.DocStoreCTE;
import com.sap.cds.impl.qat.CTE;
import com.sap.cds.impl.qat.QatAssociationNode;
import com.sap.cds.impl.qat.QatBuilder;
import com.sap.cds.impl.qat.QatElementNode;
import com.sap.cds.impl.qat.QatEntityNode;
import com.sap.cds.impl.qat.QatEntityRootNode;
import com.sap.cds.impl.qat.QatNode;
import com.sap.cds.impl.qat.QatSelectableNode;
import com.sap.cds.impl.qat.QatTraverser;
import com.sap.cds.impl.qat.QatVisitor;
import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSource;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.CqnCalculatedElementsSubstitutor;
import com.sap.cds.util.CqnStatementUtils;
import com.sap.cds.util.ElementSelector;
import com.sap.cds.util.NestedStructsResolver;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public final class CommonTableExpression implements CTE {
  private final List<PreparedCqnStmt.Parameter> parameters = new ArrayList<>();
  private final Map<String, CTE> subCTEs = new LinkedHashMap<>();
  private final String name;
  private final CqnSource source;
  private final CdsStructuredType sourceType;
  private final Context context;
  private String sql;

  public CommonTableExpression(
      Context context, CdsEntity entity, CqnSelect statement, CdsStructuredType sourceType) {
    this(context, context.getDbContext().getSqlMapping(entity).cteName(), statement, sourceType);
  }

  public CommonTableExpression(
      Context context, String name, CqnSource source, CdsStructuredType sourceType) {
    this.context = context;
    this.name = name;
    this.source = source;
    this.sourceType = sourceType;
  }

  public static Stream<String> with(
      Context context,
      Deque<QatSelectableNode> outer,
      List<PreparedCqnStmt.Parameter> params,
      Map<String, CTE> ctes) {
    collect(ctes, context, outer);
    if (ctes.isEmpty()) {
      return Stream.empty();
    }

    return with(ctes, params);
  }

  private static Stream<String> with(
      Map<String, CTE> ctes, List<PreparedCqnStmt.Parameter> params) {
    LinkedHashMap<String, String> cteSqlMap = new LinkedHashMap<>();
    cte2sql(ctes, cteSqlMap, params);
    if (cteSqlMap.isEmpty()) {
      return Stream.empty();
    }
    return Stream.of("WITH " + String.join(", ", cteSqlMap.values()));
  }

  /**
   * Recursively collects SQL for CTEs and their sub-CTEs, ensuring each is only added once and in
   * correct order (sub-CTEs before parent CTE).
   *
   * @param ctes the map of entity to CTE to process
   * @param cteSqlMap the (result) map collecting SQL strings for each CTE
   * @param params the list of parameters to collect
   */
  private static void cte2sql(
      Map<String, CTE> ctes,
      LinkedHashMap<String, String> cteSqlMap,
      List<PreparedCqnStmt.Parameter> params) {
    ctes.forEach(
        (e, cte) -> {
          if (!cteSqlMap.containsKey(e)) {
            cte2sql(cte.subCTEs(), cteSqlMap, params);
            cteSqlMap.computeIfAbsent(
                e,
                k -> {
                  params.addAll(cte.params());
                  return cte.toSQL();
                });
          }
        });
  }

  public static void collect(
      Map<String, CTE> ctes, Context context, Deque<QatSelectableNode> outer) {

    var model = context.getCdsModel();

    QatVisitor cteVisitor =
        new QatVisitor() {

          @Override
          public void visit(QatEntityRootNode root) {
            if (ctes.containsKey(root.rowType().getQualifiedName())) {
              return;
            }
            if (CdsModelUtils.isRuntimeView(model, root.rowType())) {
              ctes.putAll(resolveRuntimeViewsToCTEs(context, root.rowType()));
            } else if (hasCalculatedElements(root)) {
              String name = root.rowType().getQualifiedName();
              ctes.put(name, buildForCalculatedElements(context, root));
            }
          }

          @Override
          public void visit(QatAssociationNode assoc) {
            CdsEntity targetEntity = assoc.association().targetEntity();
            if (assoc.skipJoin() || ctes.containsKey(targetEntity.getQualifiedName())) {
              // TODO (2234) don't generate CTEs for external entities w/o table
              // Requires detection of "run with mocks" mode where external entities have a mock
              // table (see BLI 2158)
              return;
            }
            if (CdsModelUtils.isRuntimeView(model, targetEntity)) {
              ctes.putAll(resolveRuntimeViewsToCTEs(context, targetEntity));
            } else if (targetsDocStore(targetEntity)) {
              Set<String> elements = new HashSet<>();
              assoc.children().stream().map(QatNode::name).forEach(elements::add);
              targetEntity.keyElements().map(CdsElement::getName).forEach(elements::add);

              String tableName = targetEntity.getQualifiedName().replace(".", "_");
              ctes.put(assoc.alias(), new DocStoreCTE(context, tableName, assoc.alias(), elements));

            } else if (!ctes.containsKey(assoc.rowType().getQualifiedName())
                && hasCalculatedElements(assoc)) {
              ctes.put(targetEntity.getQualifiedName(), buildForCalculatedElements(context, assoc));
            }
          }

          private boolean hasCalculatedElements(QatEntityNode node) {
            CdsEntity target = node.rowType();
            // Views and anonymous types do not retain element definition
            return !target.isView()
                && !target.isAnonymous()
                && node.children().stream()
                    .anyMatch(c -> (c instanceof QatElementNode en && en.element().isCalculated()));
          }
        };

    QatTraverser.take(cteVisitor).traverse(outer.getLast());
  }

  private static Map<String, CTE> resolveRuntimeViewsToCTEs(Context context, CdsEntity view) {
    Map<String, CTE> ctes = new HashMap<>();
    resolveRuntimeViews(ctes, context.getCdsModel(), view, context);

    return ctes;
  }

  private static void resolveRuntimeViews(
      Map<String, CTE> ctes, CdsModel model, CdsEntity view, Context context) {
    if (ctes.containsKey(view.getQualifiedName())) {
      return;
    }
    CqnSelect projection =
        view.query()
            .orElseThrow(
                () ->
                    new IllegalStateException(
                        "Entity " + view.getQualifiedName() + " does not have a projection"));
    // TODO only select columns that are used in outer query
    projection =
        new ElementSelector(view, ElementSelector.NON_VIRTUAL_NON_DRAFT, false)
            .resolveStar(projection);
    CdsEntity projectionTarget = CdsModelUtils.entity(model, projection.ref());
    CqnSelect resolvedProjection =
        resolveNestedProjections(
            model, projection, projectionTarget, ElementSelector.TO_ONE_MANAGED);
    var cte = new CommonTableExpression(context, view, resolvedProjection, projectionTarget);
    ctes.put(view.getQualifiedName(), cte);
    if (CdsModelUtils.isRuntimeView(model, projectionTarget)) {
      resolveRuntimeViews(ctes, model, projectionTarget, context);
    }
  }

  private static CqnSelect resolveNestedProjections(
      CdsModel model,
      CqnSelect projection,
      CdsEntity projectionTarget,
      ElementSelector.Filter filter) {
    Select<?> query = CQL.copy((Select<?>) projection);
    NestedStructsResolver.resolveNestedStructs(query, projectionTarget, filter);
    query = CqnStatementUtils.resolveExpands(model, query, projectionTarget, filter);
    CqnStatementUtils.unfoldInline(query, projectionTarget);

    return query;
  }

  private static CommonTableExpression buildForCalculatedElements(
      Context context, QatEntityNode node) {
    CdsEntity target = node.rowType();
    SqlMapping sqlMapping = context.getDbContext().getSqlMapping(target);

    // All calculated elements selected by this node are replaced
    // with refs aliased as if they are the plain columns.
    List<CqnSelectListItem> items =
        target
            .elements()
            .filter(CdsElement::isCalculated)
            .map(e -> CQL.get(e.getName()).as(sqlMapping.columnLikeAlias(e)))
            .collect(Collectors.toList());

    items.add(select(CQL.plain(QatBuilder.ROOT_ALIAS + ".*")).build());

    // Everything that points to calculated elements replaced with expressions
    // defining them
    CqnSelect statement =
        CQL.copy(Select.from(target).columns(items), new CqnCalculatedElementsSubstitutor(target));
    return new CommonTableExpression(context, target, statement, target);
  }

  @Override
  public String toSQL() {
    render();
    return sql;
  }

  @Override
  public Map<String, CTE> subCTEs() {
    render();
    return subCTEs;
  }

  private void render() {
    if (sql == null) {
      if (source.isSelect()) {
        sql =
            SelectStatementBuilder.forCTE(
                    context, parameters, source.asSelect(), sourceType, subCTEs)
                .build()
                .sql();
      } else {
        throw new UnsupportedOperationException("Unsupported CTE source type: " + source);
      }
      sql = name + " AS (" + sql + ")";
    }
  }

  @Override
  public List<PreparedCqnStmt.Parameter> params() {
    return this.parameters;
  }

  @Override
  public String name() {
    return name;
  }
}
