/*
 * © 2018-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.impl.sql;

import static com.sap.cds.impl.sql.SQLStatementBuilder.commaSeparated;
import static com.sap.cds.impl.sql.SpaceSeparatedCollector.joining;
import static com.sap.cds.util.CdsModelUtils.entity;

import com.google.common.collect.Streams;
import com.sap.cds.SessionContext;
import com.sap.cds.impl.Context;
import com.sap.cds.impl.PreparedCqnStmt.Parameter;
import com.sap.cds.impl.builder.model.Conjunction;
import com.sap.cds.impl.localized.LocaleUtils;
import com.sap.cds.impl.qat.CTE;
import com.sap.cds.impl.qat.FromClauseBuilder;
import com.sap.cds.impl.qat.QatBuilder;
import com.sap.cds.impl.qat.QatSelectRootNode;
import com.sap.cds.impl.qat.QatSelectableNode;
import com.sap.cds.impl.qat.Ref2QualifiedColumn;
import com.sap.cds.impl.sql.collate.Collating;
import com.sap.cds.impl.sql.collate.Collator;
import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.jdbc.spi.StatementResolver;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnLock;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnSelectListValue;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnSource;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.util.CqnStatementUtils;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

public class SelectStatementBuilder implements SQLStatementBuilder {

  private final Context context;
  private final CqnSelect select;
  private final Deque<QatSelectableNode> outer;
  private final StatementResolver statementResolver;
  private final SessionContext sessionContext;
  private final List<Parameter> params;

  private Ref2QualifiedColumn aliasResolver;
  private TokenToSQLTransformer toSQL;
  private final SqlMapping sqlMapping;
  private final FromClauseBuilder fromClauseBuilder;
  private final LocaleUtils localeUtils;
  private final Collator collator;
  private final CdsEntity targetEntity;
  private final boolean isSubquery;
  private final Map<String, CTE> ctes;
  private final boolean withCTEs;

  private SelectStatementBuilder(
      Context context,
      List<Parameter> params,
      CqnSelect select,
      CdsEntity targetEntity,
      CdsStructuredType sourceType,
      Deque<QatSelectableNode> outer,
      Collator collator,
      boolean isSubquery,
      Map<String, CTE> ctes,
      boolean withCTEs) {
    this.context = context;
    this.params = params;
    this.select = select;
    this.outer = append(outer, new QatBuilder(context, select, sourceType, outer.size()).create());
    this.statementResolver = context.getDbContext().getStatementResolver();
    this.sessionContext = context.getSessionContext();
    this.ctes = ctes;
    this.withCTEs = withCTEs;
    this.fromClauseBuilder = new FromClauseBuilder(context, params, ctes, select.hints());
    this.localeUtils = new LocaleUtils(context.getCdsModel(), context.getDataStoreConfiguration());
    this.targetEntity = targetEntity;
    this.collator = collator;
    this.sqlMapping = context.getDbContext().getSqlMapping(targetEntity);
    this.isSubquery = isSubquery || outer.peekFirst() instanceof QatSelectRootNode;
  }

  public static SelectStatementBuilder forMainQuery(
      Context context, List<Parameter> params, CqnSelect select) {
    CdsEntity targetEntity = target(context, select);

    return new SelectStatementBuilder(
        context,
        params,
        select,
        targetEntity,
        sourceType(context, select),
        new ArrayDeque<>(),
        new Collating(context).collatorFor(select, targetEntity),
        false,
        new HashMap<>(),
        true);
  }

  public static SelectStatementBuilder forSubquery(
      Context context,
      List<Parameter> params,
      CqnSelect select,
      Deque<QatSelectableNode> outer,
      Map<String, CTE> ctes,
      boolean withCTEs) {
    CdsStructuredType sourceType = sourceType(context, select);
    return new SelectStatementBuilder(
        context,
        params,
        select,
        target(context, select),
        sourceType,
        outer,
        Collating.OFF,
        true,
        ctes,
        withCTEs);
  }

  public static SelectStatementBuilder forCTE(
      Context context,
      List<Parameter> params,
      CqnSelect select,
      CdsStructuredType sourceType,
      Map<String, CTE> ctes) {
    return new SelectStatementBuilder(
        context,
        params,
        select,
        target(context, select),
        sourceType,
        new ArrayDeque<>(),
        Collating.OFF,
        true,
        ctes,
        false);
  }

  private static CdsStructuredType sourceType(Context context, CqnSelect select) {
    if (select.from().isRef()) {
      return context.getCdsModel().getEntity(select.from().asRef().firstSegment());
    }
    return CqnStatementUtils.rowType(context.getCdsModel(), select.from());
  }

  @Override
  public SQLStatement build() {
    aliasResolver =
        new Ref2QualifiedColumn(context.getDbContext()::getSqlMapping, outer, localeUtils);
    // TODO propagate statement-level collation from inner to outer query
    toSQL = new TokenToSQLTransformer(context, params, aliasResolver, outer, ctes, false, collator);
    CommonTableExpression.collect(ctes, context, outer);

    List<String> snippets = new ArrayList<>();
    snippets.add("SELECT");
    distinct().forEach(snippets::add);
    if (isSubquery) {
      columnsSubquery().forEach(snippets::add);
    } else {
      columnsMainQuery().forEach(snippets::add);
    }
    from().forEach(snippets::add);
    where().forEach(snippets::add);
    groupBy().forEach(snippets::add);
    having().forEach(snippets::add);
    orderBy().forEach(snippets::add);
    limit().forEach(snippets::add);
    lock().forEach(snippets::add);
    collator.withCollation().ifPresent(snippets::add);
    if (!isSubquery) {
      hints().forEach(snippets::add);
    }
    List<Parameter> sqlParams = new ArrayList<>();
    String sql = withCTEs(snippets.stream(), sqlParams).collect(joining());

    return new SQLStatement(sql, sqlParams);
  }

  private Stream<String> withCTEs(Stream<String> snippets, List<Parameter> sqlParams) {
    if (withCTEs) {
      snippets =
          Stream.concat(CommonTableExpression.with(context, outer, sqlParams, ctes), snippets);
    }
    sqlParams.addAll(params);

    return snippets;
  }

  private Stream<String> distinct() {
    if (select.isDistinct()) {
      return Stream.of("DISTINCT");
    }

    return Stream.empty();
  }

  private Stream<String> columnsMainQuery() {
    List<CqnSelectListItem> items = select.items();
    if (items.isEmpty()) {
      throw new IllegalStateException("select * not expected");
    }

    Stream.Builder<String> sql = Stream.builder();
    items.stream()
        .flatMap(CqnSelectListItem::ofValue)
        .forEach(
            slv -> {
              sql.add(",");
              String column = toSQL.selectColumn(slv.value());
              sql.add(column);
              slv.alias().ifPresent(a -> sql.add("as").add(SQLHelper.delimited(a)));
            });

    return sql.build().skip(1);
  }

  private Stream<String> columnsSubquery() {
    List<CqnSelectListItem> items = select.items();
    if (items.isEmpty()) {
      return Stream.of(aliasResolver.rootAlias() + ".*");
    }

    Set<String> displayNames = new HashSet<>();

    Stream.Builder<String> sql = Stream.builder();
    items.stream()
        .forEach(
            sli -> {
              if (sli.isStar()) {
                sql.add(",");
                sql.add(aliasResolver.rootAlias() + ".*");
              } else if (sli instanceof CqnSelectListValue slv) {
                String col = toSQL.selectColumn(slv.value());
                if (col.isEmpty()) {
                  return; // reverse mapped association
                }
                String[] columns = col.split(", ");

                for (String c : columns) {
                  var ca = ColumnAlias.of(c, slv);
                  String column = ca.column();
                  var alias =
                      ca.alias()
                          .flatMap(
                              a -> {
                                String a2 = sqlMapping.delimitedCasing(a.replace('.', '_'));
                                return !column.endsWith(a2) ? Optional.of(a2) : Optional.empty();
                              });
                  String displayName = alias.orElse(column);
                  if (displayNames.add(displayName)) {
                    sql.add(",");
                    sql.add(column);
                    alias.ifPresent(a -> sql.add("as").add(a));
                  }
                }
              }
            });

    return sql.build().skip(1);
  }

  public record ColumnAlias(String column, Optional<String> alias) {
    // TODO refactor alias handling
    public static ColumnAlias of(String column, CqnSelectListValue slv) {
      var col = column.split("#");
      Optional<String> alias = column.endsWith("#") ? Optional.of(col[1]) : slv.alias();
      return new ColumnAlias(col[0], alias);
    }

    public static String alias(CqnElementRef ref, CqnElementRef fk) {
      return ref.alias().map(a -> "#" + fk.path().replace(fk.firstSegment(), a) + "#").orElse("");
    }
  }

  private Stream<String> from() {
    return Streams.concat(Stream.of("FROM"), fromClauseBuilder.sql(outer));
  }

  private Stream<String> where() {
    Optional<CqnPredicate> where = Conjunction.and(select.where(), select.search());
    CqnSource source = select.from();
    if (source.isRef()) {
      CqnStructuredTypeRef ref = source.asRef();
      where = Conjunction.and(where, ref.targetSegment().filter());
    }

    if (collator.usesCollate()) {
      Locale locale = sessionContext.getLocale();
      statementResolver.collate(locale).ifPresent(aliasResolver::startCollate);
    }
    Stream.Builder<String> sql = Stream.builder();
    where
        .map(w -> toSQL.toSQL(targetEntity, w))
        .ifPresent(
            s -> {
              sql.add("WHERE");
              sql.add(s);
            });
    aliasResolver.stopCollate();

    return sql.build();
  }

  private Stream<String> groupBy() {
    List<CqnValue> groupByTokens = select.groupBy();
    if (groupByTokens.isEmpty()) {
      return Stream.empty();
    }
    return Streams.concat(
        Stream.of("GROUP BY"), commaSeparated(groupByTokens.stream(), toSQL::apply));
  }

  private Stream<String> having() {
    Optional<CqnPredicate> having = select.having();

    if (collator.usesCollate()) {
      Locale locale = sessionContext.getLocale();
      statementResolver.collate(locale).ifPresent(aliasResolver::startCollate);
    }
    Stream.Builder<String> sql = Stream.builder();
    having
        .map(h -> toSQL.toSQL(targetEntity, h))
        .ifPresent(
            s -> {
              sql.add("HAVING");
              sql.add(s);
            });
    aliasResolver.stopCollate();

    return sql.build();
  }

  private Stream<String> orderBy() {
    List<CqnSortSpecification> orderBy = select.orderBy();
    if (orderBy.isEmpty()) {
      return Stream.empty();
    }
    return Streams.concat(Stream.of("ORDER BY"), commaSeparated(orderBy.stream(), toSQL::sortSpec));
  }

  private Stream<String> limit() {
    long top = select.top();
    long skip = select.skip();

    Stream.Builder<String> sql = Stream.builder();
    if (top < 0 && skip > 0) {
      top = Integer.MAX_VALUE;
    }

    if (top >= 0) {
      sql.add("LIMIT");
      sql.add(toSQL.apply(CQL.constant(top)));
    }

    if (skip > 0) {
      sql.add("OFFSET");
      sql.add(toSQL.apply(CQL.val(skip)));
    }

    return sql.build();
  }

  private Stream<String> lock() {
    Stream<CqnLock> stream = select.getLock().stream();

    return stream.flatMap(statementResolver::lockClause);
  }

  private Stream<String> hints() {
    Stream.Builder<String> sql = Stream.builder();
    statementResolver.hints(select.hints()).ifPresent(sql::add);

    return sql.build();
  }

  private static Deque<QatSelectableNode> append(
      Deque<QatSelectableNode> prefix, QatSelectableNode newRoot) {
    ArrayDeque<QatSelectableNode> roots = new ArrayDeque<>(prefix);
    roots.add(newRoot);

    return roots;
  }

  private static CdsEntity target(Context context, CqnSelect select) {
    return select.from().isRef() ? entity(context.getCdsModel(), select.from().asRef()) : null;
  }
}
