/*
 * © 2021-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.jdbc.generic;

import static java.util.stream.Collectors.joining;

import com.sap.cds.impl.parser.builder.ExpressionBuilder;
import com.sap.cds.impl.parser.token.CqnPlainImpl;
import com.sap.cds.impl.sql.SQLHelper;
import com.sap.cds.jdbc.spi.FunctionMapper;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.Predicate;
import com.sap.cds.ql.cqn.CqnPlain;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.util.DataUtils;
import java.util.List;
import java.util.Locale;

public class GenericFunctionMapper implements FunctionMapper {

  private static final String TOLOWER = "tolower";
  protected static final CqnPlain LIKE = CqnPlainImpl.plain("LIKE");
  protected static final CqnPlain ILIKE = CqnPlainImpl.plain("ILIKE");
  protected static final CqnPlain ESCAPE = CqnPlainImpl.plain("ESCAPE");
  protected static final String UNICODE_WHITESPACE_SQL =
      SQLHelper.literal(DataUtils.UNICODE_WHITESPACE);

  @Override
  public String toSql(String cqnFunc, List<String> args) {
    String func = cqnFunc.toLowerCase(Locale.US);
    return switch (func) { // NOSONAR
      case "average" -> "AVG(" + args.get(0) + ")";
      case "countdistinct" -> "COUNT(DISTINCT " + args.get(0) + ")";
      case TOLOWER -> genericFuncToSql("LOWER", args);
      case "toupper" -> genericFuncToSql("UPPER", args);
      case "current_date" -> "current_date";
      case "current_time" -> "current_time";
      case "current_timestamp" -> "current_timestamp";
      case "count" -> args.isEmpty() ? "COUNT(*)" : genericFuncToSql(cqnFunc, args);
      case "matchespattern" -> genericFuncToSql("LIKE_REGEX", args);
      case "indexof" -> genericFuncToSql("instr", args) + " - 1";
      case "trim" -> "TRIM(BOTH " + UNICODE_WHITESPACE_SQL + " FROM " + args.get(0) + ")";
      default -> genericFuncToSql(cqnFunc, args);
    };
  }

  protected String genericFuncToSql(String func, List<String> args) {
    return func.toUpperCase(Locale.US) + args.stream().collect(joining(", ", "(", ")"));
  }

  @Override
  public Predicate like(CqnValue value, CqnValue pattern, String esc, boolean caseInsensitive) {
    if (caseInsensitive) {
      value = CQL.func(TOLOWER, value);
      pattern = CQL.func(TOLOWER, pattern);
    }

    return ExpressionBuilder.create(value, LIKE, pattern, ESCAPE, CQL.constant(esc)).predicate();
  }

  protected Predicate ilike(CqnValue value, CqnValue pattern, String esc, boolean caseInsensitive) {
    CqnPlain operator = caseInsensitive ? ILIKE : LIKE;
    return ExpressionBuilder.create(value, operator, pattern, ESCAPE, CQL.constant(esc))
        .predicate();
  }
}
