/*
 * © 2020-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.jdbc.generic;

import static com.sap.cds.DataStoreConfiguration.SEARCH_MODE_LOCALIZED_ASSOC;
import static com.sap.cds.impl.localized.LocaleUtils.hasLocalizedElements;

import com.sap.cds.DataStoreConfiguration;
import com.sap.cds.jdbc.spi.SearchResolver;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import java.util.Collection;
import java.util.HashSet;
import java.util.Locale;

/**
 * A {@link SearchResolver} implementation for all generic SQL databases except HANA. It either uses
 * the localized_ view generated by the CDS compiler directly. In case each localized element can be
 * resolved by a 'localized' the search expression is resolved via this association at runtime.
 */
public class GenericSearchResolver extends AbstractSearchResolver {

  public GenericSearchResolver(DataStoreConfiguration config, CdsModel cdsModel, Locale locale) {
    super(config, cdsModel, locale);
  }

  @Override
  protected void resolve(
      CqnSelect select,
      CqnPredicate search,
      CdsStructuredType targetType,
      Collection<CqnElementRef> searchableRefs) {
    Collection<CqnElementRef> badRefs = new HashSet<>();

    boolean languageIsGiven = locale != null;
    // search non-localized view and additionally texts entity via association
    // "localized"
    final boolean resolveLocalizedViaAssociation =
        languageIsGiven
            && hasLocalizedElements(targetType, searchableRefs)
            && allLocalizedElementsAreReachableViaLocalizedAssociation(
                targetType, searchableRefs, badRefs)
            && !hasAliasedLocalizedElementsInView(targetType, searchableRefs, badRefs);

    if (!resolveLocalizedViaAssociation) {
      resolveUsingLocalizedViewWithLike(select, search, targetType, searchableRefs);
    } else {
      resolveUsingLocalizedAssociationWithLike(select, search, targetType, searchableRefs);
    }
  }

  @Override
  protected String defaultSearchMode() {
    return SEARCH_MODE_LOCALIZED_ASSOC;
  }
}
