/*
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.jdbc.h2;

import com.sap.cds.CdsDataStoreException;
import com.sap.cds.impl.sql.SQLHelper;
import com.sap.cds.impl.sql.SqlMappingImpl;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsStructuredType;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class H2SqlMapping extends SqlMappingImpl {

  private static final Pattern PATTERN =
      Pattern.compile(
          "([^\\.\\|\\[\\]]+)|(\\[(\\d+)\\])"); // group path segments and array index inside []

  public H2SqlMapping(CdsStructuredType rowType, Function<String, String> casing) {
    super(rowType, casing);
  }

  @Override
  public String jsonQuery(String contextItem, String jsonPath) {
    if (!jsonPath.startsWith("$.")) {
      throw new CdsDataStoreException(
          "Unsupported JSON path: '" + jsonPath + "' must start with '$.'");
    }
    jsonPath = jsonPath.substring(2);

    Matcher matcher = PATTERN.matcher(jsonPath);
    StringBuilder sb = new StringBuilder();
    int lastEnd = 0;

    while (matcher.find()) {
      sb.append(jsonPath, lastEnd, matcher.start());
      if (matcher.group(1) != null) {
        // quote the segment
        sb.append(SQLHelper.delimited(matcher.group(1)));
      } else if (matcher.group(2) != null) {
        // increment the index
        sb.append(Integer.parseInt(matcher.group(2)) + 1);
      }
      lastEnd = matcher.end();
    }
    sb.append(jsonPath, lastEnd, jsonPath.length());

    return "(" + contextItem + ")." + sb.toString();
  }

  @Override
  public String jsonValue(String contextItem, String jsonPath, CdsBaseType type) {
    String jsonValue = jsonQuery(contextItem, jsonPath);
    if (type == null) {
      return jsonValue;
    }
    return switch (type) {
      case STRING -> cast(jsonValue, "VARCHAR");
      case INT16 -> cast(jsonValue, "SMALLINT");
      case INTEGER, INT32 -> cast(jsonValue, "INTEGER");
      case INT64 -> cast(jsonValue, "BIGINT");
      case BOOLEAN -> cast(jsonValue, "BOOLEAN");
      case DATE -> cast(jsonValue, "DATE");
      case TIME -> cast(jsonValue, "TIME WITHOUT TIME ZONE");
      case DATETIME -> cast(jsonValue, "TIMESTAMP(0) WITH TIME ZONE");
      case TIMESTAMP -> cast(jsonValue, "TIMESTAMP(7) WITH TIME ZONE");
      default -> jsonValue;
    };
  }

  private static String cast(String jsonValue, String type) {
    return "CAST(TRIM(BOTH '\"' FROM " + jsonValue + ") AS " + type + ")";
  }
}
