/*
 * © 2022-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.jdbc.h2;

import static com.sap.cds.impl.sql.SQLHelper.commaSeparated;
import static java.util.stream.Collectors.joining;

import com.sap.cds.jdbc.generic.GenericStatementResolver;
import java.util.Optional;
import java.util.stream.Stream;

public class H2StatementResolver extends GenericStatementResolver {

  /**
   * H2: MERGE INTO
   *
   * <p>Updates existing rows, and insert rows that don't exist. If no key column is specified, the
   * primary key columns are used to find the row. If more than one row per new row is affected, an
   * exception is thrown.
   *
   * <p>https://www.h2database.com/html/commands.html#merge_into
   */
  @Override
  public String upsert(
      String table,
      Stream<String> keyColumns,
      Stream<String> upsertColumns,
      Stream<String> upsertValues) {
    String columns = commaSeparated(upsertColumns);
    String values = commaSeparated(upsertValues);

    return Stream.of("MERGE INTO", table, columns, "VALUES", values).collect(joining(" "));
  }

  @Override
  public Optional<String> timeoutClause(int timeoutSeconds) {
    if (timeoutSeconds > 0) {
      return Optional.of("WAIT " + timeoutSeconds);
    } else {
      return Optional.of("NOWAIT");
    }
  }
}
