/*
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.jdbc.hana;

import static com.sap.cds.impl.util.RealVectorUtils.toByteArray;

import com.sap.cds.CdsDataStoreException;
import com.sap.cds.CdsVector;
import com.sap.cds.impl.builder.model.RealVector;
import com.sap.cds.jdbc.generic.GenericBinder;
import java.io.InputStream;
import java.io.Reader;
import java.sql.Blob;
import java.sql.NClob;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.TimeZone;

// package private
class HanaBinder extends GenericBinder {

  // package private
  HanaBinder(TimeZone timeZone) {
    // no public construction
    super(7, timeZone);
  }

  @Override
  protected Reader getLargeString(ResultSet result, int i) throws SQLException {
    NClob nClob = result.getNClob(i);
    return null == nClob ? null : nClob.getCharacterStream();
  }

  @Override
  protected InputStream getLargeBinary(ResultSet result, int i) throws SQLException {
    Blob blob = result.getBlob(i);
    return null == blob ? null : blob.getBinaryStream();
  }

  @Override
  protected void setRealVector(PreparedStatement pstmt, int i, Object vector) throws SQLException {
    if (vector instanceof RealVector v) {
      vector = v.vector();
    }
    if (vector == null) {
      pstmt.setNull(i, Types.BINARY);
    } else if (vector instanceof byte[] b) {
      pstmt.setBytes(i, b);
    } else if (vector instanceof float[] f) {
      pstmt.setBytes(i, toByteArray(f));
    } else if (vector instanceof String s) {
      pstmt.setBytes(i, toByteArray(s));
    } else {
      throw new CdsDataStoreException(
          vector.getClass().getName() + " is not supported for type cds.Vector");
    }
  }

  @Override
  protected CdsVector getRealVector(ResultSet result, int i) throws SQLException {
    byte[] vector = result.getBytes(i);

    return RealVector.hanaRealVector(vector);
  }
}
