/*
 * © 2018-2021 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.jdbc.hana;

import com.sap.cds.jdbc.generic.GenericExceptionAnalyzer;
import java.sql.SQLException;

/**
 * The HanaExceptionAnalyzer can analyze incoming SQLExceptions from a SAP HANA connection and
 * determine by the SQL error code which problem has occurred.
 *
 * <p>All SAP HANA SQL error codes can be found here:
 * https://launchpad.support.sap.com/#/notes/0002658020
 */
public class HanaExceptionAnalyzer extends GenericExceptionAnalyzer {

  /** transaction rolled back by lock wait timeout */
  private static final int ERR_TX_ROLLBACK_LOCK_TIMEOUT = 131;

  /** Resource busy and NOWAIT specified */
  private static final int ERR_TX_LOCK_ACQUISITION_FAIL = 146;

  /** cannot insert NULL or update to NULL */
  private static final int ERR_SQL_NOT_NULL = 287;

  /** unique constraint violated */
  private static final int ERR_SQL_UNIQUE_VIOLATED = 301;

  @Override
  public boolean isUniqueConstraint(SQLException ex) {
    return ex.getErrorCode() == ERR_SQL_UNIQUE_VIOLATED;
  }

  @Override
  public boolean isNotNullConstraint(SQLException ex) {
    return ex.getErrorCode() == ERR_SQL_NOT_NULL;
  }

  @Override
  public boolean isLockTimeout(SQLException ex) {
    int errorCode = ex.getErrorCode();
    return errorCode == ERR_TX_LOCK_ACQUISITION_FAIL || errorCode == ERR_TX_ROLLBACK_LOCK_TIMEOUT;
  }
}
