/*
 * © 2021-2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.jdbc.postgresql;

import com.sap.cds.CdsDataStoreConnector.Capabilities;
import com.sap.cds.DataStoreConfiguration;
import com.sap.cds.jdbc.generic.GenericDbContext;
import com.sap.cds.jdbc.spi.ExceptionAnalyzer;
import com.sap.cds.jdbc.spi.FunctionMapper;
import com.sap.cds.jdbc.spi.PredicateMapper;
import com.sap.cds.jdbc.spi.SessionVariableSetter;
import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.jdbc.spi.StatementResolver;
import com.sap.cds.jdbc.spi.ValueBinder;
import com.sap.cds.reflect.CdsStructuredType;
import java.util.Locale;
import java.util.TimeZone;
import java.util.function.Function;

public final class PostgreSqlDbContext extends GenericDbContext {

  static final int TIMESTAMP_PRECISION = 6;

  @Override
  public FunctionMapper getFunctionMapper() {
    return new PostgreSqlFunctionMapper();
  }

  @Override
  public StatementResolver getStatementResolver() {
    return new PostgreSqlStatementResolver();
  }

  @Override
  public ExceptionAnalyzer getExceptionAnalyzer() {
    return new PostgreSqlExceptionAnalyzer();
  }

  public PostgreSqlDbContext(DataStoreConfiguration dataStoreConfiguration) {
    super(dataStoreConfiguration);
  }

  @Override
  public Capabilities getCapabilities() {
    return new Capabilities() {
      @Override
      public int timestampPrecision() {
        return TIMESTAMP_PRECISION;
      }
    };
  }

  @Override
  public ValueBinder getBinder(TimeZone timeZone) {
    return new PostgreSqlBinder(timeZone);
  }

  @Override
  public PredicateMapper getPredicateMapper() {
    return new PostgreSqlPredicateMapper();
  }

  @Override
  public SessionVariableSetter getSessionVariableSetter() {
    return new PostgreSqlSessionVariableSetter();
  }

  @Override
  public Function<String, String> casing() {
    return name -> name.toLowerCase(Locale.US);
  }

  @Override
  public SqlMapping getSqlMapping(CdsStructuredType rowType) {
    return new PostgresSqlMapping(rowType, casing());
  }
}
