/*
 * © 2024 SAP SE or an SAP affiliate company. All rights reserved.
 */
package com.sap.cds.jdbc.postgresql;

import com.sap.cds.impl.sql.SQLHelper;
import com.sap.cds.impl.sql.SqlMappingImpl;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsStructuredType;
import java.util.Arrays;
import java.util.function.Function;
import java.util.stream.Collectors;

public class PostgresSqlMapping extends SqlMappingImpl {

  public PostgresSqlMapping(CdsStructuredType rowType, Function<String, String> casing) {
    super(rowType, casing);
  }

  @Override
  public String jsonQuery(String contextItem, String jsonPath) {
    /*
     * Returns all JSON items returned by the JSON path for the specified JSON
     * value. For SQL-standard JSON path expressions it returns the JSON values
     * selected from target.
     *
     * https://www.postgresql.org/docs/current/functions-json.html
     */
    //		return "jsonb_path_query(" + contextItem + ", " + SQLHelper.literal(jsonPath) + ")"; //
    // Postgres 15+

    // https://www.postgresql.org/docs/current/functions-json.html#FUNCTIONS-JSON-PROCESSING
    return contextItem + " #> " + path(jsonPath); // Postgres 15+

    // https://www.postgresql.org/docs/current/functions-json.html#SQLJSON-QUERY-FUNCTIONS
    //		return "JSON_QUERY(" + contextItem + ", " + SQLHelper.literal(jsonPath) + ")";       //
    // Postgres >= 17.0
  }

  @Override
  public String jsonValue(String contextItem, String jsonPath, CdsBaseType type) {
    // https://www.postgresql.org/docs/current/functions-json.html#FUNCTIONS-JSON-PROCESSING
    String jsonValue = contextItem + " #>> " + path(jsonPath); // Postgres 15+
    if (type != null) {
      jsonValue = "(" + jsonValue + ")::" + sqlType(type);
    }
    return jsonValue;

    // https://www.postgresql.org/docs/current/functions-json.html#SQLJSON-QUERY-FUNCTIONS
    //		return "JSON_VALUE(" + contextItem + ", " + SQLHelper.literal(jsonPath) + " RETURNING " +
    // sqlType(type) + ")"; // PostgreSQL >= 17.0
  }

  private static String path(String jsonPath) {
    String path = Arrays.stream(jsonPath.split("\\.")).skip(1).collect(Collectors.joining(","));
    return SQLHelper.literal("{" + path + "}");
  }

  private String sqlType(CdsBaseType type) {
    return switch (type) {
      case STRING, UUID -> "varchar";
      case BOOLEAN -> "boolean";
      case INTEGER, INT32 -> "integer";
      case INT16 -> "smallint";
      case INTEGER64, INT64 -> "bigint";
      case DECIMAL -> "decimal";
      case TIMESTAMP -> "timestamp(6) at time zone 'UTC'";
      case DATETIME -> "timestamp(0) at time zone 'UTC'";
      case DATE -> "date";
      case TIME -> "time";
      case DOUBLE -> "float8";
      case LARGE_STRING -> "text";
      case BINARY -> "bytea";
      default -> "varchar";
    };
  }
}
