/************************************************************************
 * © 2021-2025 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.ql.impl;

import static com.sap.cds.impl.EntityCascader.cascadeDelete;
import static com.sap.cds.impl.EntityCascader.EntityKeys.keys;
import static com.sap.cds.impl.EntityCascader.EntityOperation.insert;
import static com.sap.cds.impl.EntityCascader.EntityOperation.nop;
import static com.sap.cds.impl.EntityCascader.EntityOperation.root;
import static com.sap.cds.impl.EntityCascader.EntityOperation.updateOrInsert;
import static com.sap.cds.impl.EntityCascader.EntityOperation.upsert;
import static com.sap.cds.impl.RowImpl.row;
import static com.sap.cds.reflect.CdsAnnotatable.byAnnotation;
import static com.sap.cds.util.CdsModelUtils.concreteKeyNames;
import static com.sap.cds.util.CdsModelUtils.isCascading;
import static com.sap.cds.util.CdsModelUtils.isReverseAssociation;
import static com.sap.cds.util.CdsModelUtils.isSingleValued;
import static com.sap.cds.util.CqnStatementUtils.hasInfixFilter;
import static com.sap.cds.util.DataUtils.generateUuidKeys;
import static com.sap.cds.util.DataUtils.isFkUpdate;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.sap.cds.CdsData;
import com.sap.cds.CdsDataStore;
import com.sap.cds.CdsList;
import com.sap.cds.Result;
import com.sap.cds.Row;
import com.sap.cds.SessionContext;
import com.sap.cds.impl.Cascader;
import com.sap.cds.impl.EntityCascader.EntityData;
import com.sap.cds.impl.EntityCascader.EntityKeys;
import com.sap.cds.impl.EntityCascader.EntityOperation;
import com.sap.cds.impl.EntityCascader.EntityOperation.Operation;
import com.sap.cds.impl.EntityCascader.EntityOperations;
import com.sap.cds.impl.builder.model.ExpressionImpl;
import com.sap.cds.impl.builder.model.StructuredTypeRefBuilder;
import com.sap.cds.impl.parser.token.RefSegmentBuilder;
import com.sap.cds.ql.CQL;
import com.sap.cds.ql.CdsDataException;
import com.sap.cds.ql.RefBuilder;
import com.sap.cds.ql.RefBuilder.RefSegment;
import com.sap.cds.ql.Select;
import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.StructuredTypeRef;
import com.sap.cds.ql.cqn.CqnEtagPredicate;
import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSelectListItem;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.cqn.CqnUpdate;
import com.sap.cds.ql.cqn.CqnUpsert;
import com.sap.cds.reflect.CdsAssociationType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.CdsModelUtils.CascadeType;
import com.sap.cds.util.CqnStatementUtils;
import com.sap.cds.util.DataUtils;
import com.sap.cds.util.OccUtils;
import com.sap.cds.util.OnConditionAnalyzer;

public class DeepUpdateSplitter {
	private static final Logger logger = LoggerFactory.getLogger(DeepUpdateSplitter.class);
	private final CdsDataStore dataStore;
	private final SessionContext session;
	private CdsEntity entity;
	private EntityOperations operations;
	private boolean deepUpsert;

	public DeepUpdateSplitter(CdsDataStore dataStore) {
		this.dataStore = dataStore;
		this.session = dataStore.getSessionContext();
	}

	public EntityOperations computeOperations(CdsEntity targetEntity, CqnUpdate update,
			Map<String, Object> targetKeys) {
		StructuredType<?> path = targetRef(update);

		return computeOps(Operation.UPDATE, path, targetEntity, targetKeys, update.entries());
	}

	public EntityOperations computeOperations(CdsEntity targetEntity, CqnUpsert upsert,
			Map<String, Object> targetKeys) {
		this.deepUpsert = true;
		StructuredType<?> path = CQL.to(RefSegmentBuilder.copy(upsert.ref().segments()));

		return computeOps(Operation.UPSERT, path, targetEntity, targetKeys, upsert.entries());
	}

	private EntityOperations computeOps(Operation op, StructuredType<?> path, CdsEntity targetEntity,
			Map<String, Object> targetKeys, List<Map<String, Object>> entries) {
		entity = targetEntity;
		operations = new EntityOperations();
		operations.entries(entries);
		List<EntityData> updateEntries = determineEntries(targetKeys, path);

		// move ETag predicate to data
		if (!updateEntries.isEmpty()) {
			moveETagToEntries(updateEntries, path);
		}

		for (EntityData entry : updateEntries) {
			EntityKeys entityId = keys(entity, entry);
			operations.add(root(entityId, op, session).update(entry));
		}
		if (!updateEntries.isEmpty()) { // compute ops for child entities
			entity.associations().forEach(assoc -> cascade(path, assoc, updateEntries));
		}

		return operations;
	}

	private void moveETagToEntries(List<EntityData> updateEntries, StructuredType<?> path) {
		OccUtils.getVersionElement(entity).ifPresent(el -> {
			Optional<CqnPredicate> filter = path.asRef().targetSegment().filter();
			Optional<CqnEtagPredicate> eTagPredicate = filter.flatMap(OccUtils::eTagPredicate);
			// existence of oldVersionParam indicates that no ETag * was present
			boolean oldVersionParam = filter.map(p -> OccUtils.containsVersionParam(p, el)).orElse(false);
			if (eTagPredicate.isPresent() || !oldVersionParam) {
				List<Object> values = eTagPredicate.map(OccUtils::concreteEtagValues).orElse(List.of());
				if (values.size() == 1) {
					Object versionValue = values.get(0);
					// set data with checked eTag value
					updateEntries.forEach(e -> e.put(el.getName(), versionValue));
				} else {
					// eTag was ambiguous or force update, can't represent in data
					updateEntries.forEach(e -> e.remove(el.getName()));
				}
			}
		});
	}

	private List<EntityData> determineEntries(Map<String, Object> targetKeys, StructuredType<?> path) {
		Set<String> keyElements = concreteKeyNames(entity);
		if (!hasInfixFilter(path.asRef())) {
			operations.entries().forEach(e -> addKeyValues(entity, targetKeys, e));
			if (entriesContainValuesFor(keyElements)) {
				// key values in data, no restricting filter
				return operations.entries().stream().map(EntityData::of).toList();
			}
		}
		Set<String> targetKeyElements = targetKeys.keySet();
		Set<String> missingKeys = Sets.filter(keyElements, k -> !targetKeyElements.contains(k));
		if (!entriesContainValuesFor(missingKeys)) {
			// missing key values (searched update) -> select w/ filter
			return selectKeyValues(path, missingKeys);
		}
		// filter condition may restrict the update
		return evaluateFilter(path, targetKeys, keyElements);
	}

	private boolean entriesContainValuesFor(Set<String> elements) {
		return operations.entries().stream().allMatch(e -> e.keySet().containsAll(elements));
	}

	private List<EntityData> selectKeyValues(StructuredType<?> path, Set<String> missingKeys) {
		if (operations.entries().size() == 1) {
			logger.warn("Update data is missing key values of entity {}. Executing query to determine key values.",
					entity.getQualifiedName());
			Select<?> select = Select.from(path).columns(missingKeys.stream().map(CQL::get));
			Result result = dataStore.execute(select);
			Map<String, Object> singleEntry = operations.entries().get(0);
			List<EntityData> updateEntries = new ArrayList<>();
			if (result.rowCount() == 1) {
				Row keyValues = result.single();
				singleEntry.putAll(keyValues);
				updateEntries.add(EntityData.of(singleEntry));
			} else if (result.rowCount() > 1) { // searched deep update
				result.forEach(row -> row.putAll(DataUtils.copyMap(singleEntry)));
				result.stream().map(EntityData::of).forEach(updateEntries::add);
			}
			return updateEntries;
		}
		throw new CdsDataException("Update data is missing key values " + missingKeys + " of entity " + entity);
	}

	private List<EntityData> evaluateFilter(StructuredType<?> path, Map<String, Object> targetKeys,
			Set<String> keyElements) {
		logger.debug("Executing query to evaluate update filter condition {}", path);
		if (!targetKeys.isEmpty()) {
			// add key values from infix filter to each data entry
			operations.entries().forEach(e -> addKeyValues(entity, targetKeys, e));
		}
		List<EntityData> updateEntries = new ArrayList<>();
		Map<String, Row> entityKeys = selectKeysMatchingPathFilter(path, keyElements, operations.entries());
		Map<String, EntityData> updateData = operations.entries().stream()
				.collect(toMap(row -> index(row, keyElements), EntityData::of));
		entityKeys.forEach((hash, key) ->
			updateEntries.add(updateData.get(hash))
		);
		logger.debug("Update filter condition fulfilled by {} entities", entityKeys.size());
		return updateEntries;
	}

	private static void addKeyValues(CdsEntity entity, Map<String, Object> keyValues, Map<String, Object> target) {
		keyValues.forEach((key, valInRef) -> {
			Object valInData = target.put(key, valInRef);
			if (valInData != null && !valInData.equals(valInRef)) {
				throw new CdsDataException("Values for key element '" + key
						+ "' in update data do not match values in update ref or where clause");
			}
		});
		target.putAll(Maps.filterValues(DataUtils.keyValues(entity, target), v -> !Objects.isNull(v)));
	}

	private static StructuredType<?> targetRef(CqnUpdate update) {
		if (!CqnStatementUtils.containsPathExpression(update.where())) {
			return CqnStatementUtils.targetRef(update);
		}
		throw new UnsupportedOperationException("Deep updates with path in where clause are not supported");
	}

	private Map<String, Row> selectKeysMatchingPathFilter(StructuredType<?> path, Set<String> keys,
			List<Map<String, Object>> keyValueSets) {
		Select<?> select = Select.from(filteredPath(path, keys)).columns(keys.stream().map(CQL::get));
		List<Map<String, Object>> params = OccUtils.addOldVersionParam(entity, keyValueSets);
		Result entries = dataStore.execute(select, params);

		return entries.stream().collect(toMap(row -> index(row, keys), Function.identity()));
	}

	private CqnStructuredTypeRef filteredPath(StructuredType<?> path, Set<String> filterParams) {
		RefBuilder<StructuredTypeRef> refBuilder = StructuredTypeRefBuilder.copy(path.asRef());
		RefSegment targetSegment = refBuilder.targetSegment();
		targetSegment.filter(ExpressionImpl.byParams(filterParams).and(targetSegment.filter().orElse(null)));

		return refBuilder.build();
	}

	private void cascade(StructuredType<?> path, CdsElement assoc, List<EntityData> entries) {
		List<EntityData> updateEntries = entries.stream().filter(e -> e.containsKey(assoc.getName())).toList();
		if (!updateEntries.isEmpty()) {
			CdsEntity targetEntity = assoc.getType().as(CdsAssociationType.class).getTarget();
			if (isSingleValued(assoc.getType())) {
				toOne(path, assoc, targetEntity, updateEntries);
			} else {
				toMany(path, assoc, targetEntity, updateEntries);
			}
		}
	}

	/*
	 * Computes operations for to-one associations / compositions - remove / delete
	 * if association is mapped to null - insert / update if association is mapped
	 * to data map
	 */
	private void toOne(StructuredType<?> path, CdsElement assoc, CdsEntity child,
			List<EntityData> parentEntries) {

		CdsEntity parent = assoc.getDeclaringType();
		boolean forward = !isReverseAssociation(assoc);
		boolean composition = assoc.getType().as(CdsAssociationType.class).isComposition();
		Set<String> parentKeys = concreteKeyNames(parent);

		Map<String, Row> expandedParentEntries = deepUpsert ? emptyMap()
				// execute batch select to determine the association's target key values
				: selectPathExpandToOneTargetKeys(path, assoc, forward, parentKeys, parentEntries);

		List<EntityData> childEntries = new ArrayList<>(parentEntries.size());
		for (EntityData parentEntry : parentEntries) {
			EntityData childEntry = EntityData.of(getDataFor(assoc, parentEntry));

			if (deepUpsert) {
				upsertToOne(parent, child, assoc, forward, composition, parentEntry, childEntry, childEntries);
			} else {
				updateToOne(parent, child, assoc, forward, composition, parentKeys, expandedParentEntries, parentEntry,
						childEntry, childEntries);
			}
		}
		// cascade over associations
		child.associations().forEach(a -> cascade(path.to(assoc.getName()), a, childEntries));
	}

	private void upsertToOne(CdsEntity parent, CdsEntity child, CdsElement assoc, boolean forward, boolean composition,
							 EntityData parentEntry, EntityData childEntry, List<EntityData> childEntries) {
		if (childEntry == null) {
			// clear FKs & delete target when cascading
			remove(assoc, forward, parent, singletonList(parentEntry), assoc.getName());

		} else if (isCascading(CascadeType.INSERT, assoc) || isCascading(CascadeType.UPDATE, assoc)) {
			childEntries.add(childEntry);
			// TODO allow to delete old child via annotation
			operations.add(xsert(parentEntry, assoc, forward, child, childEntry, false));
		} else if (forward && !composition) {
			// set ref to existing target
			removeNonFkValues(assoc, childEntry);
		}
	}

	private void updateToOne(CdsEntity parentEntity, CdsEntity targetEntity, CdsElement assoc, boolean forward,
			boolean composition, Set<String> parentKeys, Map<String, Row> expandedParentEntries,
							 EntityData parentUpdateEntry, EntityData targetUpdateEntry,
			List<EntityData> targetUpdateEntries) {
		Row parentEntry = expandedParentEntries.getOrDefault(index(parentUpdateEntry, parentKeys), row(emptyMap()));
		Map<String, Object> targetEntry = getDataFor(assoc, parentEntry);

		if (targetEntry != null && !targetEntry.isEmpty()) {
			// assoc points to an existing target entity
			if (targetUpdateEntry == null) {
				// clear FKs & delete target when cascading
				remove(assoc, forward, targetEntity, singletonList(targetEntry), null);

			} else if (targetChange(assoc, forward && !composition, targetEntity, targetEntry, targetUpdateEntry)) {
				// target key values changed
				if (isCascading(CascadeType.INSERT, assoc) || isCascading(CascadeType.UPDATE, assoc)) {
					// composition -> insert / assoc -> update or insert
					operations.add(
							xsert(parentUpdateEntry, assoc, forward, targetEntity, targetUpdateEntry, composition));
					targetUpdateEntries.add(targetUpdateEntry);
				} else { // update FK (relaxed data)
					removeNonFkValues(assoc, targetUpdateEntry);
				}
				if (composition) { // delete old target
					delete(targetEntity, targetEntry, null);
				}

			} else if (isCascading(CascadeType.UPDATE, assoc)) { // update target
				operations.add(update(targetEntity, targetEntry, targetUpdateEntry));
				targetUpdateEntries.add(targetUpdateEntry);
			}
			// same target, not cascading update -> ignore

		} else if (targetUpdateEntry != null) { // assoc is null
			if (forward && !composition && !isCascading(CascadeType.INSERT, assoc)) {
				// set ref to existing target
				removeNonFkValues(assoc, targetUpdateEntry);
			} else { // insert or update target
				boolean generatedKey = generateUuidKeys(targetEntity, targetUpdateEntry.data());
				if (forward && generatedKey) { // update parent with target ref (FK) values
					Map<String, Object> parentKeyValues = keys(parentEntity, parentUpdateEntry);
					Map<String, Object> targetRefValues = fkValues(assoc, false, targetUpdateEntry);
					operations.add(update(parentEntity, parentKeyValues, EntityData.ofFKs(targetRefValues)));
				}
				boolean insertOnly = composition || generatedKey;
				operations.add(xsert(parentUpdateEntry, assoc, forward, targetEntity, targetUpdateEntry, insertOnly));
				targetUpdateEntries.add(targetUpdateEntry);
			}
		}
	}

	private static boolean targetChange(CdsElement assoc, boolean forward, CdsEntity targetEntity,
			Map<String, Object> oldEntry, Map<String, Object> newEntry) {
		Set<String> refElements = forward ? refElements(assoc, forward) : concreteKeyNames(targetEntity);
		return refElements.stream().anyMatch(k -> {
			Object newVal = newEntry.get(k);
			return newVal != null && !newVal.equals(oldEntry.get(k));
		});
	}

	private static void removeNonFkValues(CdsElement assoc, Map<String, Object> data) {
		Set<String> assocKeys = refElements(assoc, true);
		data.keySet().retainAll(assocKeys);
	}

	private Map<String, Row> selectPathExpandToOneTargetKeys(StructuredType<?> path, CdsElement assoc,
			boolean forwardMapped, Set<String> parentKeys, List<EntityData> keyValueSets) {
		logger.debug("Executing query to determine target entity of {}", assoc.getQualifiedName());
		List<CqnSelectListItem> slis = parentKeys.stream().map(CQL::get).collect(toList());
		Set<String> refElements = refElements(assoc, forwardMapped);
		refElements.addAll(CdsModelUtils.targetKeys(assoc));
		slis.add(CQL.to(assoc.getName()).expand(refElements.toArray(new String[refElements.size()])));
		CqnSelect select = Select.from(filteredPath(path, parentKeys)).columns(slis);
		List<Map<String, Object>> list = keyValueSets.stream().map(EntityData::withFKs).toList();
		Iterable<Map<String, Object>> params = OccUtils.addOldVersionParam(entity, list);
		Result targetEntries = dataStore.execute(select, params);

		return targetEntries.stream().collect(toMap(row -> index(row, parentKeys), Function.identity()));
	}

	private Map<String, Object> fkValues(CdsElement assoc, boolean reverseMapped, EntityData data) {
		return new OnConditionAnalyzer(assoc, reverseMapped, session).getFkValues(data.withFKs());
	}

	/*
	 * Computes operations for to-many associations / compositions - insert / update
	 * entities that are in the update data list - remove / delete entities that are
	 * not in the update data list
	 */
	private void toMany(StructuredType<?> path, CdsElement assoc, CdsEntity targetEntity,
			List<EntityData> parentEntries) {
		boolean composition = assoc.getType().as(CdsAssociationType.class).isComposition();
		Set<String> parentKeys = concreteKeyNames(assoc.getDeclaringType());
		Set<String> targetKeys = CdsModelUtils.targetKeys(assoc);
		// execute batch select to determine the association target entities of
		// non-delta entries
		Map<String, Row> fullSetTargetEntries = selectTargetEntries(assoc, parentKeys, targetKeys, parentEntries);

		List<EntityData> updateEntries = new ArrayList<>(parentEntries.size());
		for (EntityData parentEntry : parentEntries) {
			List<Map<String, Object>> targetUpdateEntries = getDataFor(assoc, parentEntry);
			if (targetUpdateEntries == null) {
				throw new CdsDataException("Value for to-many association '" + assoc.getDeclaringType() + "." + assoc
						+ "' must not be null.");
			}
			Map<String, Object> parentRefValues = fkValues(assoc, true, parentEntry);
			if (parentRefValues.containsValue(null)) {
				throw new CdsDataException("Values of ref elements " + parentRefValues.keySet() + " for mapping "
						+ targetEntity + " to " + assoc.getDeclaringType() + " cannot be determined from update data.");
			}
			boolean delta = isDelta(targetUpdateEntries);
			for (Map<String, Object> entry : targetUpdateEntries) {
				EntityData updateEntry = EntityData.of(entry);
				updateEntry.putFKs(parentRefValues);
				EntityOperation operation;
				if (delta) { // DELTA: remove/delete via annotation
					if (isRemove(updateEntry)) {
						remove(assoc, false, targetEntity, singletonList(updateEntry.withFKs()), null);
						continue;
					}
					operation = xsert(assoc, targetEntity, updateEntry, false);

				} else { // FULL-SET: remove/delete if not present in update data
					boolean targetPresent = fullSetTargetEntries.remove(index(updateEntry.withFKs(), targetKeys)) != null;
					if (deepUpsert) {
						// Do DB upsert to deal with concurrent inserts/deletes
						operation = xsert(assoc, targetEntity, updateEntry, false);
					} else {
						boolean generatedKey = !targetPresent && generateUuidKeys(targetEntity, updateEntry.data());
						EntityKeys targetId = keys(targetEntity, updateEntry);
						if (generatedKey) { // insert
							operation = EntityOperation.insert(targetId, updateEntry, session);
						} else if (targetPresent) { // nop or update
							operation = nop(targetId, null, session).update(updateEntry, emptyMap());
						} else { // assoc: insert or update / composition: insert
							operation = xsert(assoc, targetEntity, updateEntry, composition);
						}
					}
				}
				assertCascading(operation, assoc);
				operations.add(operation);
				updateEntries.add(updateEntry);
			}
		}
		// full-sets: remove/delete all entities that are not included in the update
		// data list
		remove(assoc, false, targetEntity, fullSetTargetEntries.values(), null);

		// cascade over associations
		targetEntity.associations().forEach(a -> cascade(path.to(assoc.getName()), a, updateEntries));
	}

	private static boolean isDelta(List<Map<String, Object>> entries) {
		if (entries instanceof CdsList list) {
			return list.isDelta();
		}
		return false;
	}

	private static boolean isRemove(EntityData entry) {
		if (entry.data() instanceof CdsData data) {
			return data.isForRemoval();
		}
		return false;
	}

	private Map<String, Row> selectTargetEntries(CdsElement assoc, Set<String> parentKeys, Set<String> targetKeys,
			List<EntityData> keyValueSets) {
		logger.debug("Executing query to determine target entity of {}", assoc.getQualifiedName());
		StructuredType<?> path = CQL.entity(assoc.getDeclaringType().getQualifiedName()).filterByParams(parentKeys)
				.to(assoc.getName());
		CqnSelect select = Select.from(path).columns(targetKeys.stream().map(CQL::get));
		Result targetEntries = dataStore.execute(select, keyValueSets.stream()
				.filter(entry -> !isDelta(getDataFor(assoc, entry)))
				.map(EntityData::withFKs).toList());

		return targetEntries.stream().collect(toMap(row -> index(row, targetKeys), Function.identity()));
	}

	private EntityOperation update(CdsEntity target, Map<String, Object> entry, EntityData updateData) {
		EntityOperation op = nop(keys(target, entry), null, entry, session).update(updateData);
		updateData.putAll(op.targetKeys());

		return op;
	}

	private void remove(CdsElement association, boolean forwardMapped, CdsEntity targetEntity,
			Collection<? extends Map<String, Object>> keyValues, String path) {
		if (isCascading(CascadeType.DELETE, association)) {
			keyValues.forEach(k -> delete(targetEntity, k, path));
		} else if (!forwardMapped) {
			assertCascading(CascadeType.UPDATE, association);
			Set<String> parentRefElements = refElements(association, forwardMapped);
			keyValues.forEach(k ->
				operations.add(nop(keys(targetEntity, k), path, session).updateToNull(parentRefElements))
			);
		}
	}

	private void delete(CdsEntity entity, Map<String, Object> data, String path) {
		EntityKeys rootKey = keys(entity, data);
		CdsEntity target = entity;
		if (path != null) {
			target = entity.getTargetOf(path);
		}
		boolean cascaded = Cascader.create(CascadeType.DELETE, target).from(path)
				.cascade(p -> operations.add(EntityOperation.delete(rootKey, p.asRef().path(), session)));
		if (!cascaded) {
			// fallback for cyclic models and subqueries in where
			// compute object graph (select key values)
			cascadeDelete(dataStore, rootKey, path).forEach(operations::add);
		}
		operations.add(EntityOperation.delete(rootKey, path, session));
	}

	private EntityOperation xsert(EntityData parentData, CdsElement association, boolean forwardMapped,
			CdsEntity entity, EntityData entityData, boolean insertOnly) {
		if (!forwardMapped) { // set parent ref (FK) in target entity
			entityData.putFKs(fkValues(association, true, parentData));
		}
		return xsert(association, entity, entityData, insertOnly);
	}

	private EntityOperation xsert(CdsElement association, CdsEntity entity, EntityData entityData, boolean insertOnly) {
		EntityKeys targetEntity = keys(entity, entityData);
		boolean insert = isCascading(CascadeType.INSERT, association);
		boolean update = !insertOnly && isCascading(CascadeType.UPDATE, association);
		if (!deepUpsert && insert && update && containsStream(entityData)) {
			// InputStreams can only be consumed once
			// Determine if target exists to decide between insert and update
			CqnSelect select = Select.from(entity).columns(CQL.plain("1").as("1")).matching(targetEntity);
			update = dataStore.execute(select).rowCount() > 0;
			insert = !update;
		}
		if (insert && update) {
			if (deepUpsert || !hasDefaultValues(entity, entityData)) {
				return upsert(targetEntity, entityData, session);
			}
			return updateOrInsert(targetEntity, entityData, session);
		}
		if (insert) {
			return insert(targetEntity, entityData, session);
		}
		if (update) {
			return nop(targetEntity, null, session).update(entityData);
		}
		if (CdsModelUtils.managedToOne(association.getType()) && isFkUpdate(association, entityData, session)) {
			return nop(targetEntity, null, session);
		}
		CdsEntity target = association.getType().as(CdsAssociationType.class).getTarget();
		throw new CdsDataException(
				"UPSERT entity '%s' via association '%s.%s' is not allowed. The association does not cascade insert or update.".formatted(
				target, association.getDeclaringType(), association));
	}

	@SuppressWarnings("unchecked")
	private static boolean containsStream(Map<String, Object> entityData) {
		return entityData.values().stream()
				.anyMatch(v -> v instanceof InputStream || v instanceof Map m && containsStream(m));
	}

	private static boolean hasDefaultValues(CdsEntity entity, Map<String, Object> entityData) {
		boolean defaultValue = entity.concreteNonAssociationElements().filter(e -> e.defaultValue().isPresent())
				.anyMatch(e -> !entityData.containsKey(e.getName()));
		return defaultValue || entity.concreteNonAssociationElements().filter(byAnnotation("cds.on.insert"))
				.anyMatch(e -> !entityData.containsKey(e.getName()));
	}

	private static void assertCascading(EntityOperation op, CdsElement association) {
		switch (op.operation()) {
		case INSERT:
			assertCascading(CascadeType.INSERT, association);
			break;
		case UPDATE:
			assertCascading(CascadeType.UPDATE, association);
			break;
		case DELETE:
			assertCascading(CascadeType.DELETE, association);
			break;
		case UPDATE_OR_INSERT:
		case UPSERT:
			assertCascading(CascadeType.UPDATE, association);
			assertCascading(CascadeType.INSERT, association);
		}
	}

	private static void assertCascading(CascadeType cascadeType, CdsElement association) {
		if (!isCascading(cascadeType, association)) {
			CdsEntity target = association.getType().as(CdsAssociationType.class).getTarget();
			throw new CdsDataException(
					"%s entity '%s' via association '%s.%s' is not allowed. The association does not cascade %s.".formatted(
					cascadeType.name(), target, association.getDeclaringType(), association, cascadeType));
		}
	}

	private static Set<String> refElements(CdsElement assoc, boolean forwardMapped) {
		HashMap<String, String> mapping = new HashMap<>();
		new OnConditionAnalyzer(assoc, !forwardMapped).getFkMapping().forEach((fk, val) -> {
			if (val.isRef() && !val.asRef().firstSegment().startsWith("$")) {
				mapping.put(fk, val.asRef().lastSegment());
			}
		});
		if (forwardMapped) {
			return new HashSet<>(mapping.values());
		}
		return new HashSet<>(mapping.keySet());
	}

	public static <T> T getDataFor(CdsElement assoc, Map<String, Object> data) {
		return DataUtils.getOrDefault(data, assoc.getName(), null);
	}

	public static String index(Map<String, Object> values, Set<String> keys) {
		return values.entrySet().stream().filter(e -> keys.contains(e.getKey()))
				.map(e -> e.getKey() + ":" + e.getValue()).sorted().collect(joining("-"));
	}

}
