/************************************************************************
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.impl;

import java.util.ArrayList;

import com.sap.cds.DataStoreConfiguration;
import com.sap.cds.impl.docstore.DocStoreSelectStatementBuilder;
import com.sap.cds.impl.docstore.DocStoreUpdateStatementBuilder;
import com.sap.cds.impl.docstore.DocStoreUtils;
import com.sap.cds.impl.sql.DeleteStatementBuilder;
import com.sap.cds.impl.sql.InsertStatementBuilder;
import com.sap.cds.impl.sql.SQLStatementBuilder.SQLStatement;
import com.sap.cds.impl.sql.SelectStatementBuilder;
import com.sap.cds.impl.sql.UpdateStatementBuilder;
import com.sap.cds.impl.sql.UpsertStatementBuilder;
import com.sap.cds.ql.cqn.CqnStatement;

public class JdbcDataSourceAdapter implements SQLDataSourceAdapter {

	private final Context context;
	private final DocStoreUtils docStoreUtils;

	public JdbcDataSourceAdapter(Context context) {
		this.context = context;
		this.docStoreUtils = new DocStoreUtils(context.getCdsModel());
	}

	@Override
	public SQLStatement process(CqnStatement statement) {
		if (docStoreUtils.targetsDocStore(statement)) {
			return toHanaDocStoreSQL(statement);
		}
		return toSQL(statement);
	}

	private SQLStatement toHanaDocStoreSQL(CqnStatement statement) {
		assertThatDocStoreSupportIsEnabled();
		Context dsCtx = DocStoreUtils.wrap(context);

		if (statement.isSelect()) {
			return new DocStoreSelectStatementBuilder(dsCtx, statement.asSelect()).build();
		}

		if (statement.isInsert()) {
			return DocStoreUtils.getDocStoreInsertStatement(statement.asInsert(), dsCtx);
		}

		if (statement.isUpdate()) {
			return new DocStoreUpdateStatementBuilder(dsCtx, statement.asUpdate()).build();
		}

		if (statement.isDelete()) {
			return new DeleteStatementBuilder(dsCtx, statement.asDelete()).build();
		}

		throw new UnsupportedOperationException("Unsupported statement type: " + statement);
	}

	private SQLStatement toSQL(CqnStatement statement) {
		if (statement.isSelect()) {
			return SelectStatementBuilder.forMainQuery(context, new ArrayList<>(), statement.asSelect()).build();
		}
		if (statement.isInsert()) {
			return new InsertStatementBuilder(context, statement.asInsert()).build();
		}
		if (statement.isUpsert()) {
			return new UpsertStatementBuilder(context, statement.asUpsert()).build();
		}
		if (statement.isUpdate()) {
			return new UpdateStatementBuilder(context, statement.asUpdate()).build();
		}
		if (statement.isDelete()) {
			return new DeleteStatementBuilder(context, statement.asDelete()).build();
		}
		throw new UnsupportedOperationException("Unsupported statement type: " + statement);
	}

	public void assertThatDocStoreSupportIsEnabled() {
		if (!context.getDataStoreConfiguration().getProperty(DataStoreConfiguration.DOCSTORE_INTEGRATION_ENABLED,
				false)) {
			throw new UnsupportedOperationException(
					"Can't handle an entity annotated as doc-store enabled: DocStore integration is not enabled.");
		}
	}
}
