/************************************************************************
 * © 2019-2025 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.impl.qat;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Strings;
import com.sap.cds.impl.builder.model.ElementRefImpl;
import com.sap.cds.impl.docstore.DocStoreUtils;
import com.sap.cds.impl.localized.LocaleUtils;
import com.sap.cds.impl.sql.SqlMappingImpl;
import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.ql.cqn.CqnElementRef;
import com.sap.cds.ql.cqn.CqnExistsSubquery;
import com.sap.cds.ql.cqn.CqnReference.Segment;
import com.sap.cds.ql.cqn.CqnValidationException;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.OnConditionAnalyzer;

public class Ref2QualifiedColumn implements Ref2Column {

	private final QatSelectableNode root;
	private final QatSelectableNode outer;
	private final String rootName;
	private final SqlMapping sqlMapping;
	private final LocaleUtils localeUtils;
	private String collateClause = null;

	public Ref2QualifiedColumn(Function<CdsStructuredType, SqlMapping> mapping, Deque<QatSelectableNode> outer, LocaleUtils localeUtils) {
		Iterator<QatSelectableNode> reverse = outer.descendingIterator();
		this.root = qatRoot(reverse.next());
		this.rootName = root.rowType().getQualifiedName();
		this.outer = reverse.hasNext() ? reverse.next() : null;
		this.sqlMapping = mapping.apply(root.rowType());
		this.localeUtils = localeUtils;
	}

	public void startCollate(String collateClause) {
		if (Strings.emptyToNull(collateClause) != null) {
			this.collateClause = " " + collateClause;
		} else {
			stopCollate();
		}
	}

	public void stopCollate() {
		this.collateClause = null;
	}

	public Stream<String> apply(CqnElementRef ref) {
		return apply(Clause.WHERE, ref);
	}

	@Override
	public Stream<String> apply(Clause clause, CqnElementRef ref) {
		List<? extends Segment> segments = ref.segments();
		CdsBaseType refType = ref.type().map(CdsBaseType::cdsType).orElse(null);
		if (ref.firstSegment().equals(CqnExistsSubquery.OUTER)) {
			return qualifiedColumnName(clause, segments.subList(1, segments.size()), outer, refType);
		}

		return qualifiedColumnName(clause, segments, root, refType);
	}

	private Stream<String> qualifiedColumnName(Clause clause, List<? extends Segment> segments, QatNode node, CdsBaseType type) {
		boolean firstSegment = segments.size() > 1;
		List<String> columnNames = new ArrayList<>(2);
		String jsonPath = null;

		for (int i = 0; i < segments.size(); i++) {
			Segment s = segments.get(i);
			if (firstSegment && s.id().equals(rootName)) {
				continue;
			}
			if (node instanceof QatElementNode en && en.isJson()) {
				jsonPath = segments.stream().skip(i).map(Segment::id).collect(Collectors.joining(".", "$.", ""));
				break;
			}

			node = node.child(s.id(), s.filter());
			assertNotNull(segments, node);
			firstSegment = false;
		}

		if (node instanceof QatElementNode en) {
			columnNames.add(columnName(en));
		} else if (node instanceof QatAssociationNode an) {
			QatAssociation association = an.association();
			if (CdsModelUtils.isReverseAssociation(root.rowType().getAssociation(association.name()))) {
				throw new CqnValidationException("The association '" + node.name() + "' is mapped via an inverse FK and therefore not supported on the select list");
			}
			new OnConditionAnalyzer(association.name(), association.on(), false).getFkMapping().keySet()
					.stream().map(sqlMapping::columnName).forEach(columnNames::add);
		} else {
			throw new CqnValidationException(node.name() + " does not refer to an element");
		}
		do {
			node = node.parent();
		} while (node != null && !(node instanceof QatSelectableNode));
		assertNotNull(segments, node);

		var tableAlias = ((QatSelectableNode) node).alias();
		var columns = columnNames.stream().map(c -> column(tableAlias, c));
		if (!Strings.isNullOrEmpty(jsonPath)) {
			var jsPath = jsonPath;
			if (type != null && type != CdsBaseType.MAP) {
				columns = columns.map(col -> sqlMapping.jsonValue(col, jsPath, type));
			} else {
				columns = columns.map(col -> switch (clause) {
					case WHERE, ORDERBY -> sqlMapping.jsonValue(col, jsPath, type);
					case SELECT -> sqlMapping.jsonQuery(col, jsPath);
				});
			}
		}
		return columns;
	}

	private static String column(String alias, String col) {
		return alias + "." + col;
	}

	private static QatSelectableNode qatRoot(QatNode root) {
		Optional<QatNode> next;
		while ((next = childInSource(root)).isPresent()) {
			root = next.get();
		}

		return (QatSelectableNode) root;
	}

	private static Optional<QatNode> childInSource(QatNode root) {
		return root.children().stream().filter(QatNode::inSource).findFirst();
	}

	private String columnName(QatElementNode elementNode) {
		CdsElement cdsElement = elementNode.element();
		String columnName = sqlMapping.columnName(cdsElement);
		if (elementNode.parent() instanceof QatStructuredElementNode) {
			if (DocStoreUtils.targetsDocStore((CdsStructuredType) cdsElement.getDeclaringType())) {
				return columnName;
			}
			if (cdsElement.findAnnotation(SqlMappingImpl.CDS_PERSISTENCE_NAME).isEmpty()) {
				columnName = structuredColumnName(elementNode);
			}
		}
		// TODO extract COLLATE from column name resolver (issues/1071)
		if (collateClause != null && localeUtils.requiresCollate(cdsElement)) {
			columnName = columnName + collateClause;
		}

		return columnName;
	}

	private String structuredColumnName(QatNode node) {
		List<String> elementName = new ArrayList<>();
		elementName.add(((QatElementNode) node).element().getName());
		QatNode parent = node.parent();
		while (parent instanceof QatStructuredElementNode) {
			QatStructuredElementNode structNode = (QatStructuredElementNode) parent;
			elementName.add(structNode.element().getName());
			parent = structNode.parent();
		}
		Collections.reverse(elementName);
		return sqlMapping.delimitedCasing(String.join("_", elementName));
	}

	private static void assertNotNull(List<? extends Segment> segments, QatNode node) {
		if (node == null) {
			CqnElementRef ref = ElementRefImpl.elementRef(segments, null, null);
			throw new CqnValidationException("Unresolvable path expression: " + ref.toJson());
		}
	}
	
	public String rootAlias() {
		return root.alias();
	}

}
