/*********************************************************************
 * (C) 2025 SAP SE or an SAP affiliate company. All rights reserved. *
 *********************************************************************/
package com.sap.cds.impl.sql.collate;

import java.util.Locale;
import static com.sap.cds.ql.impl.SelectBuilder.COLLATING;
import static com.sap.cds.ql.impl.SelectBuilder.COLLATING_OFF;
import java.util.Optional;

import com.sap.cds.SessionContext;
import com.sap.cds.impl.Context;
import com.sap.cds.impl.localized.LocaleUtils;
import com.sap.cds.jdbc.spi.StatementResolver;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnValue;
import com.sap.cds.reflect.CdsBaseType;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.util.CqnStatementUtils;

public class Collating {

	public static final Collator OFF = new Collator() {
	};

	private final SessionContext sessionContext;
	private final StatementResolver statementResolver;
	private final LocaleUtils localeUtils;
	private final Locale locale;

	public Collating(Context context) {
		sessionContext = context.getSessionContext();
		statementResolver = context.getDbContext().getStatementResolver();
		localeUtils = new LocaleUtils(context.getCdsModel(), context.getDataStoreConfiguration());
		locale = sessionContext.getLocale();
	}

	public Collator collatorFor(CqnSelect select, CdsStructuredType targetEntity) {
		if (locale == null || hasCollatingOffHint(select)) {
			return OFF;
		}

		return statementResolver.supportsStatementWideCollation() && localeUtils.requiresCollationClause(select, locale)
				? new StatementWide(select)
				: new Collate(targetEntity);
	}

	private static boolean hasCollatingOffHint(CqnSelect select) {
		return COLLATING_OFF.equals(select.hints().get(COLLATING));
	}

	private class Collate implements Collator {
		private final CdsStructuredType targetEntity;

		private Collate(CdsStructuredType targetEntity) {
			this.targetEntity = targetEntity;
		}

		@Override
		public Optional<String> collate(CqnSortSpecification o) {
			if (requiresCollate(o.value())) {
				return statementResolver.collate(o, sessionContext.getLocale());
			} else {
				return Optional.empty();
			}
		}

		@Override
		public boolean usesCollate() {
			return true;
		}

		private boolean requiresCollate(CqnValue value) {
			if (targetEntity == null) {
				return false;
			}
			if (value.isRef()) {
				return localeUtils.requiresCollate(targetEntity, value.asRef());
			}
			return CqnStatementUtils.getCdsType(targetEntity, value).filter(t -> CdsBaseType.STRING == t).isPresent();
		}

	}

	private class StatementWide implements Collator {
		private final CqnSelect select;

		private StatementWide(CqnSelect select) {
			this.select = select;
		}

		@Override
		public Optional<String> withCollation() {
			return statementResolver.statementWideCollation(select, locale);
		}
	}
}
