/************************************************************************
 * © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.jdbc.generic;

import static com.sap.cds.impl.localized.LocaleUtils.isLocalized;

import java.util.Locale;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.jdbc.spi.TableNameResolver;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsStructuredType;

/**
 * A {@link TableNameResolver} that resolves localized strings using
 * locale-unspecific views using session variables
 */
public class LocaleVariableTableNameResolver implements TableNameResolver {
	private static final Logger logger = LoggerFactory.getLogger(LocaleVariableTableNameResolver.class);

	private final Function<CdsStructuredType, SqlMapping> mapping;
	private final Locale locale;

	public LocaleVariableTableNameResolver(Function<CdsStructuredType, SqlMapping> mapping, Locale locale) {
		this.mapping = mapping;
		this.locale = locale;
	}

	@Override
	public String tableName(CdsEntity entity) {
		SqlMapping sqlMapping = mapping.apply(entity);
		String entityName = entity.getQualifiedName();
		if (locale != null && isLocalized(entity)) {
			String localizedViewName = sqlMapping.localizedViewName();
			if (logger.isDebugEnabled()) {
				logger.debug("Resolving {} to localized view {} with locale '{}'",
						entityName, localizedViewName, locale.toLanguageTag());
			}

			return localizedViewName;
		}
		String tableName = sqlMapping.tableName();
		logger.debug("Resolving {} to {}", entityName, tableName);

		return tableName;
	}
}
