/************************************************************************
 * © 2024-2025 SAP SE or an SAP affiliate company. All rights reserved. *
 ************************************************************************/
package com.sap.cds.jdbc.hana.hierarchies;

import java.util.ArrayDeque;
import java.util.List;
import java.util.OptionalInt;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.sap.cds.impl.Context;
import com.sap.cds.impl.PreparedCqnStmt.Parameter;
import com.sap.cds.impl.sql.SQLStatementBuilder.SQLStatement;
import com.sap.cds.impl.sql.SelectStatementBuilder;
import com.sap.cds.impl.sql.TokenToSQLTransformer;
import com.sap.cds.jdbc.spi.SqlMapping;
import com.sap.cds.jdbc.spi.TableFunctionMapper;
import com.sap.cds.jdbc.spi.TableNameResolver;
import com.sap.cds.ql.cqn.CqnSelect;
import com.sap.cds.ql.cqn.CqnSortSpecification;
import com.sap.cds.ql.cqn.CqnSource;
import com.sap.cds.ql.cqn.CqnStructuredTypeRef;
import com.sap.cds.ql.hana.CqnHierarchy;
import com.sap.cds.ql.hana.CqnHierarchyGenerator;
import com.sap.cds.ql.hana.CqnHierarchySubset;
import com.sap.cds.reflect.CdsEntity;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.CdsStructuredType;
import com.sap.cds.util.CdsModelUtils;
import com.sap.cds.util.CqnStatementUtils;

public abstract class HierarchyFunctionMapper<T extends CqnHierarchy> implements TableFunctionMapper {

	private static final String SOURCE = "SOURCE";
	private static final String LPAREN = "(";
	private static final String RPAREN = ")";

	protected final Context context;
	protected final T hierarchy;
	protected final List<Parameter> params;
	protected final TokenToSQLTransformer toSQL;
	protected final CdsModel model;

	private HierarchyFunctionMapper(Context context, List<Parameter> params, T hierarchy) {
		this.context = context;
		this.model = context.getCdsModel();
		this.hierarchy = hierarchy;
		this.params = params;
		this.toSQL = toSQL(context, params, hierarchy.source());
	}

	private static TokenToSQLTransformer toSQL(Context context, List<Parameter> params, CqnSource source) {
		if (source.isRef()) {
			CqnStructuredTypeRef ref = source.asRef();
			CdsEntity entity = CdsModelUtils.entity(context.getCdsModel(), ref);
			SqlMapping sqlMapping = context.getDbContext().getSqlMapping(entity);
			String tableName = sqlMapping.tableName();

			return TokenToSQLTransformer.notCollating(context, (clause, r) -> Stream.of(sqlMapping.columnName(r)), entity, tableName, params);
		} else if (source.isSelect()) {
			CdsStructuredType rowType = CqnStatementUtils.rowType(context.getCdsModel(), source.asSelect());
			SqlMapping sqlMapping = context.getDbContext().getSqlMapping(rowType);

			return TokenToSQLTransformer.notCollating(context, (clause, r) -> Stream.of(sqlMapping.columnName(r)), source.asSelect(), rowType,
					params);
		} else if (source.isTableFunction()) {
			return toSQL(context, params, source.asTableFunction().source());
		} else {
			throw new IllegalStateException();
		}
	}

	@Override
	public String toSQL() {
		StringBuilder sql = new StringBuilder(hierarchy.name());
		sql.append(LPAREN);
		sql.append(SOURCE);
		sql.append(" ");
		CqnSource source = hierarchy.source();
		if (source.isRef()) {
			CqnStructuredTypeRef ref = source.asRef();
			TableNameResolver resolver = context.getTableNameResolver();
			CdsEntity entity = CdsModelUtils.entity(context.getCdsModel(), ref);
			sql.append(resolver.tableName(entity));
		} else if (source.isSelect()) {
			sql.append(LPAREN);
			CqnSelect subquery = source.asSelect();
			SQLStatement stmt = SelectStatementBuilder.forSubquery(context, params, subquery, new ArrayDeque<>()).build();
			sql.append(stmt.sql());
			sql.append(RPAREN);
		} else if (source instanceof CqnHierarchy inner) {
			String innerSQL = create(context, params, inner).toSQL();
			sql.append(innerSQL);
		} else {
			throw new IllegalStateException();
		}

		appendClauses(sql);

		sql.append(RPAREN);
		return new SQLStatement(sql.toString(), List.of()).sql();
	}

	protected CdsStructuredType rowType() {
		return CqnStatementUtils.rowType(model, hierarchy);
	}

	abstract void appendClauses(StringBuilder builder);

	@SuppressWarnings("unchecked")
	public static <T extends CqnHierarchy> HierarchyFunctionMapper<T> create(Context context, List<Parameter> params,
			T hierarchy) {
		if (hierarchy.isGenerator()) {
			return (HierarchyFunctionMapper<T>) new Generator(context, params, hierarchy.asGenerator());
		} else if (hierarchy.isHierarchySubset()) {
			return (HierarchyFunctionMapper<T>) new HierarchySubset(context, params, hierarchy.asHierarchySubset());
		} else {
			throw new IllegalStateException();
		}
	}

	private static class Generator extends HierarchyFunctionMapper<CqnHierarchyGenerator> {

		public Generator(Context context, List<Parameter> params, CqnHierarchyGenerator hierarchy) {
			super(context, params, hierarchy);
		}

		private static final String SIBL_ORDER_BY = "SIBLING ORDER BY";
		private static final String DEPTH = "DEPTH";

		@Override
		void appendClauses(StringBuilder sql) {
			List<CqnSortSpecification> siblingOrderBy = hierarchy.orderBy();
			if (!siblingOrderBy.isEmpty()) {
				appendSiblingOrderBy(sql, siblingOrderBy);
			}

			if (hierarchy.depth() != null) {
				sql.append(" ").append(DEPTH).append(" ").append(hierarchy.depth());
			}

			hierarchy.startWhere().map(w -> toSQL.toSQL(rowType(), w)).ifPresent(s -> 
				sql.append(" START WHERE ").append(s));

			sql.append(" NO CACHE");
		}

		private void appendSiblingOrderBy(StringBuilder sql, List<CqnSortSpecification> siblingOrderBy) {
			sql.append(" ").append(SIBL_ORDER_BY).append(" ");

			sql.append(siblingOrderBy.stream().map(this::sort).collect(Collectors.joining(", ")));
		}

		private String sort(CqnSortSpecification o) {
			return toSQL.sortSpec(o);
		}
	}

	private static class HierarchySubset extends HierarchyFunctionMapper<CqnHierarchySubset> {

		public HierarchySubset(Context context, List<Parameter> params, CqnHierarchySubset hierarchyNavigation) {
			super(context, params, hierarchyNavigation);
		}

		@Override
		void appendClauses(StringBuilder sql) {
			hierarchy.startWhere().map(w -> toSQL.toSQL(rowType(), w))
					.ifPresent(s -> sql.append(" START WHERE ").append(s));

			// distance
			OptionalInt from;
			OptionalInt to;
			if (hierarchy.isAncestors()) {
				// -MAX_VALUE and MIN_VALUE both indicate unlimited  
				from = hierarchy.from() <= -Integer.MAX_VALUE ? OptionalInt.empty() : OptionalInt.of(hierarchy.from());
				to = hierarchy.to() == 0 ? OptionalInt.empty() : OptionalInt.of(hierarchy.to());
			} else {
				from = hierarchy.from() == 0 ? OptionalInt.empty() : OptionalInt.of(hierarchy.from());
				to = hierarchy.to() == Integer.MAX_VALUE ? OptionalInt.empty() : OptionalInt.of(hierarchy.to());
			}
			if (from.isPresent() || to.isPresent()) {
				sql.append(" DISTANCE");
				from.ifPresent(i -> sql.append(" FROM " + i));
				to.ifPresent(i -> sql.append(" TO " + i));
			}
		}
	}

}
