/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.olingo.client.api.communication;

import org.apache.http.Header;
import org.apache.http.StatusLine;
import org.apache.olingo.commons.api.ex.ODataError;
import org.apache.olingo.commons.api.ex.ODataRuntimeException;

import java.io.InputStream;

/**
 * Represents a client error in OData.
 *
 * @see ODataError
 */
public class ODataClientErrorException extends ODataRuntimeException {

  private static final long serialVersionUID = -2551523202755268162L;

  private final StatusLine statusLine;

  private final ODataError error;

  private final InputStream rawResponse;

  private Header[] headerInfo;

  /**
   * Constructor.
   *
   * @param statusLine request status info.
   */
  public ODataClientErrorException(final StatusLine statusLine) {
    this(statusLine, null, null);
  }

  /**
   * Constructor
   *
   * @param statusLine request status info.
   * @param rawResponse raw response of the request.
   */
  public ODataClientErrorException(final StatusLine statusLine, final InputStream rawResponse) {
    this(statusLine, null, rawResponse);
  }

  /**
   * Constructor
   *
   * @param statusLine request status info.
   * @param error OData error to be wrapped.
   */
  public ODataClientErrorException(final StatusLine statusLine, final ODataError error) {
    this(statusLine, error, null);
  }

  /**
   * Constructor.
   *
   * @param statusLine request status info.
   * @param error OData error to be wrapped.
   * @param rawResponse raw response of the request.
   */
  public ODataClientErrorException(final StatusLine statusLine, final ODataError error, final InputStream rawResponse) {
    super(error == null ?
            statusLine.toString() :
            (error.getCode() == null || error.getCode().isEmpty() ? "" : "(" + error.getCode() + ") ")
                    + error.getMessage() + " [" + statusLine.toString() + "]");

    this.statusLine = statusLine;
    this.error = error;
    this.rawResponse = rawResponse;
  }

  /**
   * Gets request status info.
   *
   * @return request status info.
   */
  public StatusLine getStatusLine() {
    return statusLine;
  }

  /**
   * Gets OData error.
   *
   * @return OData error.
   */
  public ODataError getODataError() {
    return error;
  }

  /**
   * Sets headers
   * @param headerInfo
   */
  public void setHeaderInfo(Header[] headerInfo) {
    this.headerInfo = headerInfo;
  }

  /**
   * Returns headers
   * @return Header[]
   */
  public Header[] getHeaderInfo() {
    return headerInfo;
  }

  /**
   * Return raw response from the request (can be null).
   *
   * @return raw response from the request (can be null).
   */
  public InputStream getRawResponse() {
    return rawResponse;
  }
}
