/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.olingo.server.core.uri.parser;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.olingo.commons.api.edm.Edm;
import org.apache.olingo.commons.api.edm.EdmAnnotation;
import org.apache.olingo.commons.api.edm.EdmEntityType;
import org.apache.olingo.commons.api.edm.EdmFunction;
import org.apache.olingo.commons.api.edm.EdmKeyPropertyRef;
import org.apache.olingo.commons.api.edm.EdmNavigationProperty;
import org.apache.olingo.commons.api.edm.EdmParameter;
import org.apache.olingo.commons.api.edm.EdmPrimitiveType;
import org.apache.olingo.commons.api.edm.EdmPrimitiveTypeException;
import org.apache.olingo.commons.api.edm.EdmPrimitiveTypeKind;
import org.apache.olingo.commons.api.edm.EdmProperty;
import org.apache.olingo.commons.api.edm.EdmStructuredType;
import org.apache.olingo.commons.api.edm.EdmType;
import org.apache.olingo.commons.api.edm.EdmTypeDefinition;
import org.apache.olingo.commons.api.edm.FullQualifiedName;
import org.apache.olingo.commons.api.edm.annotation.EdmExpression;
import org.apache.olingo.commons.api.edm.constants.EdmTypeKind;
import org.apache.olingo.server.api.OData;
import org.apache.olingo.server.api.uri.UriParameter;
import org.apache.olingo.server.api.uri.UriResource;
import org.apache.olingo.server.api.uri.UriResourceEntitySet;
import org.apache.olingo.server.api.uri.UriResourceKind;
import org.apache.olingo.server.api.uri.UriResourceNavigation;
import org.apache.olingo.server.api.uri.UriResourcePartTyped;
import org.apache.olingo.server.api.uri.queryoption.AliasQueryOption;
import org.apache.olingo.server.api.uri.queryoption.CustomQueryOption;
import org.apache.olingo.server.api.uri.queryoption.SystemQueryOptionKind;
import org.apache.olingo.server.api.uri.queryoption.expression.Expression;
import org.apache.olingo.server.api.uri.queryoption.expression.Literal;
import org.apache.olingo.server.core.ODataImpl;
import org.apache.olingo.server.core.uri.UriParameterImpl;
import org.apache.olingo.server.core.uri.UriResourceTypedImpl;
import org.apache.olingo.server.core.uri.UriResourceWithKeysImpl;
import org.apache.olingo.server.core.uri.parser.UriTokenizer.TokenKind;
import org.apache.olingo.server.core.uri.queryoption.AliasQueryOptionImpl;
import org.apache.olingo.server.core.uri.queryoption.expression.LiteralImpl;
import org.apache.olingo.server.core.uri.validator.UriValidationException;
import org.apache.olingo.server.core.uri.validator.UriValidationException.MessageKeys;

public class ParserHelper {

  private static final OData odata = new ODataImpl();
  private static final String REST = "REST";

  protected static final Map<TokenKind, EdmPrimitiveTypeKind> tokenToPrimitiveType;
  static {
    /* Enum and null are not present in the map. These have to be handled differently. */
    Map<TokenKind, EdmPrimitiveTypeKind> temp = new EnumMap<>(TokenKind.class);
    temp.put(TokenKind.BooleanValue, EdmPrimitiveTypeKind.Boolean);
    temp.put(TokenKind.StringValue, EdmPrimitiveTypeKind.String);
    // Very large integer values are of type Edm.Decimal but this is handled elsewhere.
    temp.put(TokenKind.IntegerValue, EdmPrimitiveTypeKind.Int64);
    temp.put(TokenKind.GuidValue, EdmPrimitiveTypeKind.Guid);
    temp.put(TokenKind.DateValue, EdmPrimitiveTypeKind.Date);
    temp.put(TokenKind.DateTimeOffsetValue, EdmPrimitiveTypeKind.DateTimeOffset);
    temp.put(TokenKind.TimeOfDayValue, EdmPrimitiveTypeKind.TimeOfDay);
    temp.put(TokenKind.DecimalValue, EdmPrimitiveTypeKind.Decimal);
    temp.put(TokenKind.DoubleValue, EdmPrimitiveTypeKind.Double);
    temp.put(TokenKind.DurationValue, EdmPrimitiveTypeKind.Duration);
    temp.put(TokenKind.BinaryValue, EdmPrimitiveTypeKind.Binary);

    temp.put(TokenKind.GeographyPoint, EdmPrimitiveTypeKind.GeographyPoint);
    temp.put(TokenKind.GeometryPoint, EdmPrimitiveTypeKind.GeometryPoint);
    temp.put(TokenKind.GeographyLineString, EdmPrimitiveTypeKind.GeographyLineString);
    temp.put(TokenKind.GeometryLineString, EdmPrimitiveTypeKind.GeometryLineString);
    temp.put(TokenKind.GeographyPolygon, EdmPrimitiveTypeKind.GeographyPolygon);
    temp.put(TokenKind.GeometryPolygon, EdmPrimitiveTypeKind.GeometryPolygon);
    temp.put(TokenKind.GeographyMultiPoint, EdmPrimitiveTypeKind.GeographyMultiPoint);
    temp.put(TokenKind.GeometryMultiPoint, EdmPrimitiveTypeKind.GeometryMultiPoint);
    temp.put(TokenKind.GeographyMultiLineString, EdmPrimitiveTypeKind.GeographyMultiLineString);
    temp.put(TokenKind.GeometryMultiLineString, EdmPrimitiveTypeKind.GeometryMultiLineString);
    temp.put(TokenKind.GeographyMultiPolygon, EdmPrimitiveTypeKind.GeographyMultiPolygon);
    temp.put(TokenKind.GeometryMultiPolygon, EdmPrimitiveTypeKind.GeometryMultiPolygon);
    temp.put(TokenKind.GeographyCollection, EdmPrimitiveTypeKind.GeographyCollection);
    temp.put(TokenKind.GeometryCollection, EdmPrimitiveTypeKind.GeometryCollection);

    tokenToPrimitiveType = Collections.unmodifiableMap(temp);
  }

  protected static void requireNext(UriTokenizer tokenizer, final TokenKind required) throws UriParserException {
    if (!tokenizer.next(required)) {
      throw new UriParserSyntaxException("Expected token '" + required.toString() + "' not found.",
          UriParserSyntaxException.MessageKeys.SYNTAX);
    }
  }

  protected static void requireTokenEnd(UriTokenizer tokenizer) throws UriParserException {
    requireNext(tokenizer, TokenKind.EOF);
  }

  protected static boolean bws(UriTokenizer tokenizer) {
    return tokenizer.nextWhitespace();
  }
  
  protected static TokenKind next(UriTokenizer tokenizer, final TokenKind... kinds) {
    for (final TokenKind kind : kinds) {
      if (tokenizer.next(kind)) {
        return kind;
      }
    }
    return null;
  }

  protected static TokenKind nextPrimitiveValue(UriTokenizer tokenizer) {
    return next(tokenizer,
        TokenKind.NULL,
        TokenKind.BooleanValue,
        TokenKind.StringValue,

        // The order of the next seven expressions is important in order to avoid
        // finding partly parsed tokens (counter-intuitive as it may be, even a GUID may start with digits ...).
        TokenKind.GuidValue,
        TokenKind.DoubleValue,
        TokenKind.DecimalValue,
        TokenKind.DateTimeOffsetValue,
        TokenKind.DateValue,
        TokenKind.TimeOfDayValue,
        TokenKind.IntegerValue,

        TokenKind.DurationValue,
        TokenKind.BinaryValue,
        TokenKind.EnumValue,

        // Geography and geometry literals are defined to be primitive,
        // although they contain several parts with their own meaning.
        TokenKind.GeographyPoint,
        TokenKind.GeometryPoint,
        TokenKind.GeographyLineString,
        TokenKind.GeometryLineString,
        TokenKind.GeographyPolygon,
        TokenKind.GeometryPolygon,
        TokenKind.GeographyMultiPoint,
        TokenKind.GeometryMultiPoint,
        TokenKind.GeographyMultiLineString,
        TokenKind.GeometryMultiLineString,
        TokenKind.GeographyMultiPolygon,
        TokenKind.GeometryMultiPolygon,
        TokenKind.GeographyCollection,
        TokenKind.GeometryCollection);
  }

  protected static List<UriParameter> parseFunctionParameters(UriTokenizer tokenizer,
      final Edm edm, final EdmType referringType, final boolean withComplex,
      final Map<String, AliasQueryOption> aliases)
      throws UriParserException, UriValidationException {
    return parseParameters(tokenizer, edm, referringType, withComplex, true, aliases);
  }

  protected static List<UriParameter> parseFunctionParameters(UriTokenizer tokenizer,
      final Edm edm, final Map<String, AliasQueryOption> aliases, List<CustomQueryOption> customQueryOptions)
      throws UriParserException, UriValidationException {

    // NB: This is not always the case for the tokenizer to observe the complete
    // URL. Signature of this method serves as an additional protection against
    // accidental misuse
    if (tokenizer.next(TokenKind.EOF)) {
      // Services MAY in addition allow implicit parameter aliases for function
      // imports and for functions that are the last path segment of the URL.
      return parseImplicitParameters(aliases, customQueryOptions);
    } else {
      return parseParameters(tokenizer, edm, null, false, false, aliases);
    }
  }

  private static List<UriParameter> parseParameters(UriTokenizer tokenizer,
      final Edm edm, final EdmType referringType, final boolean withComplex,
      boolean allowCollections, final Map<String, AliasQueryOption> aliases)
      throws UriParserException, UriValidationException {
    List<UriParameter> parameters = new ArrayList<>();
    Set<String> parameterNames = new HashSet<>();
    ParserHelper.requireNext(tokenizer, TokenKind.OPEN);
    if (tokenizer.next(TokenKind.CLOSE)) {
      return parameters;
    }
    do {
      ParserHelper.requireNext(tokenizer, TokenKind.ODataIdentifier);
      final String name = tokenizer.getText();
      consumeAndCheckParameter(name, parameterNames);
      ParserHelper.requireNext(tokenizer, TokenKind.EQ);
      if (tokenizer.next(TokenKind.COMMA) || tokenizer.next(TokenKind.CLOSE) || tokenizer.next(TokenKind.EOF)) {
        throw new UriParserSyntaxException("Parameter value expected.", UriParserSyntaxException.MessageKeys.SYNTAX);
      }
      UriParameterImpl parameter = new UriParameterImpl().setName(name);
      if (tokenizer.next(TokenKind.ParameterAliasName)) {
        final String aliasName = tokenizer.getText();
        parameter.setAlias(aliasName)
            .setExpression(aliases.containsKey(aliasName) ? aliases.get(aliasName).getValue() : null);
      } else if (tokenizer.next(TokenKind.jsonArrayOrObject)) {
        if (withComplex) {
          parameter.setText(tokenizer.getText());
        } else {
          throw new UriParserSemanticException("A JSON array or object is not allowed as parameter value.",
              UriParserSemanticException.MessageKeys.COMPLEX_PARAMETER_IN_RESOURCE_PATH, tokenizer.getText());
        }
      } else if (withComplex) {
        final Expression expression = new ExpressionParser(edm, odata)
            .parse(tokenizer, referringType, null, aliases, allowCollections);
        parameter.setText(expression instanceof Literal ?
            "null".equals(((Literal) expression).getText()) ? null : ((Literal) expression).getText() :
            null)
            .setExpression(expression instanceof Literal ? null : expression);
      } else if (nextPrimitiveValue(tokenizer) == null) {
        throw new UriParserSemanticException("Wrong parameter value.",
            UriParserSemanticException.MessageKeys.INVALID_KEY_VALUE, "");
      } else {
        final String literalValue = tokenizer.getText();
        parameter.setText("null".equals(literalValue) ? null : literalValue);
      }
      parameters.add(parameter);
    } while (tokenizer.next(TokenKind.COMMA));
    ParserHelper.requireNext(tokenizer, TokenKind.CLOSE);
    return parameters;
  }

  private static List<UriParameter> parseImplicitParameters(Map<String, AliasQueryOption> aliases,
    List<CustomQueryOption> customQueryOptions)
    throws UriParserSemanticException, UriValidationException {

    List<UriParameter> result = new ArrayList<>();
    Set<String> parameterNames = new HashSet<>();
    // An implicit parameter alias is the parameter name, optionally preceded by an at (@) sign.
    // Olingo treats this as an alias -> so that what we emulate here: as if the parameter
    // is expressed with explicit alias.
    for (Entry<String, AliasQueryOption> a: aliases.entrySet()) {
      // So that such aliases treated as non-prefixed custom query options
      String parameterName = a.getKey().substring(1);
      consumeAndCheckParameter(parameterName, parameterNames);
      AliasQueryOption alias = a.getValue();
      result.add(new UriParameterImpl()
          .setName(parameterName)
          .setExpression(alias.getValue())
          .setText(alias.getText())
          .setAlias(alias.getName()));
    }

    // $apply must have functions, but they are not the last segment
    if (customQueryOptions != null) {
      for (CustomQueryOption q : customQueryOptions) {
        consumeAndCheckParameter(q.getName(), parameterNames);
        if (SystemQueryOptionKind.isSystemQueryOption(q.getName())) {
          // If a parameter name is identical to a system query option name (without the
          // optional $ prefix), the parameter name MUST be prefixed with an at (@) sign.
          throw new UriValidationException(
              "Parameter with name {} must be prefixed with '@'",
              MessageKeys.UNSUPPORTED_PARAMETER, q.getName());
        }
        String text = "null".equals(q.getText()) ? null : q.getText();
        result.add(new UriParameterImpl().setName(q.getName()).setText(text));
      }
    }
    return result;
  }

  private static void consumeAndCheckParameter(String name, Set<String> parameterNames)
      throws UriParserSemanticException {
    if (parameterNames.contains(name)) {
      throw new UriParserSemanticException("Duplicated function parameter " + name,
          UriParserSemanticException.MessageKeys.INVALID_KEY_VALUE, name);
    }
    parameterNames.add(name);
  }

  /*
   * Returns whether the provided parameter is optional.<br>
   * 
   * <br>
   * Possible values for optional:
   * <ul>
   * <li><code>@Core.OptionalParameter}</code></li>
   * <li><code>@Core.OptionalParameter: true}</code></li>
   * <li><code>@Core.OptionalParameter: {DefaultValue: 'foo'}</code></li>
   * </ul>
   *
   * Possible value for mandatory:
   * <ul>
   * <li><code>@Core.OptionalParameter: false</code></li>
   * </ul><br>
   * 
   * @param parameter the parameter to check
   * 
   * @return {@code true} if the parameter is optional, {@code false} otherwise
   */
  private static boolean isOptional(EdmParameter parameter) {
    Optional<EdmAnnotation> optionalParameter = parameter.getAnnotations().stream()
        .filter(a -> a.getTerm() != null && "Core.OptionalParameter"
            .equals(a.getTerm().getFullQualifiedName().getFullQualifiedNameAsString()))
        .findAny();
    if (!optionalParameter.isPresent()) {
      return false;
    }
    EdmExpression expression = optionalParameter.get().getExpression();
    if (expression != null && expression.isConstant()
        && "false".equalsIgnoreCase(expression.asConstant().getValueAsString())) {
      return false;
    }
    return true;
  }

  protected static void validateFunctionParameters(final EdmFunction function, final List<UriParameter> parameters,
      final Edm edm, final EdmType referringType, final Map<String, AliasQueryOption> aliases)
      throws UriParserException, UriValidationException {
    for (final UriParameter parameter : parameters) {
      final String parameterName = parameter.getName();
      final EdmParameter edmParameter = function.getParameter(parameterName);
      final boolean isNullable = edmParameter.isNullable();
      if (parameter.getText() == null && parameter.getExpression() == null && !isNullable
          && !isOptional(edmParameter)) {
        if (parameter.getAlias() == null) {
          // No alias, value is explicitly null.
          throw new UriValidationException("Missing non-nullable parameter " + parameterName,
              UriValidationException.MessageKeys.MISSING_PARAMETER, parameterName);
        } else {
          final String valueForAlias = aliases.containsKey(parameter.getAlias()) ?
              parseAliasValue(parameter.getAlias(),
                  edmParameter.getType(), edmParameter.isNullable(), edmParameter.isCollection(),
                  edm, referringType, aliases).getText() :
              null;
          // Alias value is missing or explicitly null.
          if (valueForAlias == null) {
            throw new UriValidationException("Missing alias for " + parameterName,
                UriValidationException.MessageKeys.MISSING_ALIAS, parameter.getAlias());
          }
        }
      }
    }
  }

  protected static void validateFunctionParameters(final EdmFunction function, final List<UriParameter> parameters,
      final Edm edm, final EdmType referringType)
      throws UriParserException, UriValidationException {
    // Non-nullable parameters must be provided
    Set<EdmParameter> nonNullable = function.getParameterNames().stream()
        .map(n -> function.getParameter(n))
        .filter(p -> !p.isNullable() && !p.getName().equals(function.getEntitySetPath())).collect(Collectors.toSet());
    for (EdmParameter nn : nonNullable) {
      parameters.stream().filter(p -> p.getName().equals(nn.getName())).findAny()
          .orElseThrow(() -> new UriValidationException("Missing non-nullable parameter " + nn.getName(),
              UriValidationException.MessageKeys.MISSING_PARAMETER, nn.getName()));
    }

    for (final UriParameter parameter : parameters) {
      final String parameterName = parameter.getName();
      final EdmParameter edmParameter = function.getParameter(parameterName);
      if (null == edmParameter) {
        throw new UriValidationException("Invalid named parameter " + function.getName() + "." + parameterName,
            UriValidationException.MessageKeys.INVALID_VALUE_FOR_PROPERTY, parameterName);
      }
      final boolean isNullable = edmParameter.isNullable();
      if (parameter.getText() == null && parameter.getExpression() == null && !isNullable) {
        throw new UriValidationException("Missing non-nullable parameter " + parameterName,
            UriValidationException.MessageKeys.MISSING_PARAMETER, parameterName);
      }
    }
  }

  protected static AliasQueryOption parseAliasValue(final String name, final EdmType type, final boolean isNullable,
      final boolean isCollection, final Edm edm, final EdmType referringType,
      final Map<String, AliasQueryOption> aliases) throws UriParserException, UriValidationException {
    final EdmTypeKind kind = type == null ? null : type.getKind();
    final AliasQueryOption alias = aliases.get(name);
    if (alias != null && alias.getText() != null) {
      UriTokenizer aliasTokenizer = new UriTokenizer(alias.getText());
      if (kind == null
          || !((isCollection || kind == EdmTypeKind.COMPLEX || kind == EdmTypeKind.ENTITY ?
          aliasTokenizer.next(TokenKind.jsonArrayOrObject) :
          nextPrimitiveTypeValue(aliasTokenizer, (EdmPrimitiveType) type, isNullable, null))
          && aliasTokenizer.next(TokenKind.EOF))) {
        // The alias value is not an allowed literal value, so parse it again as expression.
        aliasTokenizer = new UriTokenizer(alias.getText());
        // Don't pass on the current alias to avoid circular references.
        Map<String, AliasQueryOption> aliasesInner = new HashMap<>(aliases);
        aliasesInner.remove(name);
        final Expression expression = new ExpressionParser(edm, odata)
            .parse(aliasTokenizer, referringType, null, aliasesInner);
        final EdmType expressionType = ExpressionParser.getType(expression);
        if (aliasTokenizer.next(TokenKind.EOF)
            && (expressionType == null || type == null || expressionType.equals(type))) {
          ((AliasQueryOptionImpl) alias).setAliasValue(expression);
        } else {
          throw new UriParserSemanticException("Illegal value for alias '" + alias.getName() + "'.",
              UriParserSemanticException.MessageKeys.UNKNOWN_PART, alias.getText());
        }
      }
    }
    return alias;
  }

  protected static List<UriParameter> parseNavigationKeyPredicate(UriTokenizer tokenizer,
      final EdmNavigationProperty navigationProperty,
      final Edm edm, final EdmType referringType, final Map<String, AliasQueryOption> aliases, String protocolType)
      throws UriParserException, UriValidationException {
	  if (protocolType != null && protocolType.equalsIgnoreCase(REST)) {
		  if (tokenizer.next(TokenKind.OPEN)) {
			  throw new UriParserSyntaxException("Unexpected start of resource-path segment.",
	    	          UriParserSyntaxException.MessageKeys.SYNTAX);
		  }
	  } else {
		  if (tokenizer.next(TokenKind.OPEN)) {
		      if (navigationProperty.isCollection()) {
		        return parseKeyPredicate(tokenizer, navigationProperty.getType(), 
		        		navigationProperty.getPartner(),edm, referringType, aliases);
		      } else {
		        throw new UriParserSemanticException("A key is not allowed on "
		        		+ "non-collection navigation properties.",
		            UriParserSemanticException.MessageKeys.KEY_NOT_ALLOWED);
		      }
		    }
	  }
    
    return null;
  }

  protected static List<UriParameter> parseKeyPredicate(UriTokenizer tokenizer, final EdmEntityType edmEntityType,
      final EdmNavigationProperty partner,
      final Edm edm, final EdmType referringType, final Map<String, AliasQueryOption> aliases)
      throws UriParserException, UriValidationException {
    final List<EdmKeyPropertyRef> keyPropertyRefs = edmEntityType.getKeyPropertyRefs();
    if (tokenizer.next(TokenKind.CLOSE)) {
      throw new UriParserSemanticException(
          "Expected " + keyPropertyRefs.size() + " key predicates but got none.",
          UriParserSemanticException.MessageKeys.WRONG_NUMBER_OF_KEY_PROPERTIES,
          Integer.toString(keyPropertyRefs.size()), "0");
    }
    List<UriParameter> keys = new ArrayList<>();
    Map<String, String> referencedNames = new HashMap<>();

    if (partner != null) {
      // Prepare list of potentially missing keys to be filled from referential constraints.
      for (final String name : edmEntityType.getKeyPredicateNames()) {
        final String referencedName = partner.getReferencingPropertyName(name);
        if (referencedName != null) {
          referencedNames.put(name, referencedName);
        }
      }
      for (final EdmKeyPropertyRef candidate : keyPropertyRefs) {
    	  final UriParameter simpleKey = simpleKey(tokenizer, candidate, edm, referringType, aliases);
          if (simpleKey != null) {
            keys.add(simpleKey);
          }
      }
    }

    if (keyPropertyRefs.size() - referencedNames.size() == 1) {
      for (final EdmKeyPropertyRef candidate : keyPropertyRefs) {
       if (referencedNames.get(candidate.getName()) == null) {
          final UriParameter simpleKey = simpleKey(tokenizer, candidate, edm, referringType, aliases);
          if (simpleKey != null) {
            keys.add(simpleKey);
          }
          break;
        }
      }
    }
    if (keys.isEmpty()) {
      if (tokenizer.next(TokenKind.ODataIdentifier)) {
        keys.addAll(compoundKey(tokenizer, edmEntityType, edm, referringType, aliases, null));
      } else {
        throw new UriParserSemanticException("The key value is not valid.",
            UriParserSemanticException.MessageKeys.INVALID_KEY_VALUE, "");
      }
    }

    if (keys.size() < keyPropertyRefs.size() && partner != null) {
      // Fill missing keys from referential constraints.
      for (final String name : edmEntityType.getKeyPredicateNames()) {
        boolean found = false;
        for (final UriParameter key : keys) {
          if (name.equals(key.getName())) {
            found = true;
            break;
          }
        }
        if (!found && referencedNames.get(name) != null) {
          keys.add(0, new UriParameterImpl().setName(name).setReferencedProperty(referencedNames.get(name)));
        }
      }
    }

    // Check that all key predicates are filled from the URI.
    if (keys.size() < keyPropertyRefs.size()) {
      throw new UriParserSemanticException(
          "Expected " + keyPropertyRefs.size() + " key predicates but found " + keys.size() + ".",
          UriParserSemanticException.MessageKeys.WRONG_NUMBER_OF_KEY_PROPERTIES,
          Integer.toString(keyPropertyRefs.size()), Integer.toString(keys.size()));
    } else {
      return keys;
    }
  }

  private static UriParameter simpleKey(UriTokenizer tokenizer, final EdmKeyPropertyRef edmKeyPropertyRef,
      final Edm edm, final EdmType referringType, final Map<String, AliasQueryOption> aliases)
      throws UriParserException, UriValidationException {
    final EdmProperty edmProperty = edmKeyPropertyRef == null ? null : edmKeyPropertyRef.getProperty();
    if (nextPrimitiveTypeValue(tokenizer,
        edmProperty == null ? null : (EdmPrimitiveType) edmProperty.getType(),
        edmProperty == null ? false : edmProperty.isNullable(), null)) {
      final String literalValue = tokenizer.getText();
      ParserHelper.requireNext(tokenizer, TokenKind.CLOSE);
      return createUriParameter(edmProperty, edmKeyPropertyRef.getName(), literalValue, edm, referringType, aliases);
    } else {
      return null;
    }
  }

  protected static List<UriParameter> compoundKey(UriTokenizer tokenizer, final EdmEntityType edmEntityType,
      final Edm edm, final EdmType referringType, final Map<String, AliasQueryOption> aliases, String protocolType)
      throws UriParserException, UriValidationException {

    List<UriParameter> parameters = new ArrayList<>();
    List<String> parameterNames = new ArrayList<>();

    // To validate that each key predicate is exactly specified once, we use a list to pick from.
    List<String> remainingKeyNames = new ArrayList<>(edmEntityType.getKeyPredicateNames());

    // At least one key predicate is mandatory.  Try to fetch all.
    boolean hasComma = false;
    do {
      final String keyPredicateName = tokenizer.getText();
      if (parameterNames.contains(keyPredicateName)) {
        throw new UriValidationException("Duplicated key property " + keyPredicateName,
            UriValidationException.MessageKeys.DOUBLE_KEY_PROPERTY, keyPredicateName);
      }
      if (remainingKeyNames.isEmpty()) {
        throw new UriParserSemanticException("Too many key properties.",
            UriParserSemanticException.MessageKeys.WRONG_NUMBER_OF_KEY_PROPERTIES,
            Integer.toString(parameters.size()), Integer.toString(parameters.size() + 1));
      }
      if (!remainingKeyNames.remove(keyPredicateName)) {
        throw new UriValidationException("Unknown key property " + keyPredicateName,
            UriValidationException.MessageKeys.INVALID_KEY_PROPERTY, keyPredicateName);
      }
      parameters.add(keyValuePair(tokenizer, keyPredicateName, edmEntityType, 
    		  edm, referringType, aliases, protocolType));
      parameterNames.add(keyPredicateName);
      hasComma = tokenizer.next(TokenKind.COMMA);
      if (hasComma) {
        ParserHelper.requireNext(tokenizer, TokenKind.ODataIdentifier);
      }
    } while (hasComma);
    if (protocolType != null && protocolType.equalsIgnoreCase(REST)) {
    ParserHelper.requireNext(tokenizer, TokenKind.EOF);
    } else {
    ParserHelper.requireNext(tokenizer, TokenKind.CLOSE);
    }

    return parameters;
  }

  private static UriParameter keyValuePair(UriTokenizer tokenizer,
      final String keyPredicateName, final EdmEntityType edmEntityType,
      final Edm edm, final EdmType referringType, final Map<String, AliasQueryOption> aliases, String protocolType)
      throws UriParserException, UriValidationException {
    final EdmKeyPropertyRef keyPropertyRef = edmEntityType.getKeyPropertyRef(keyPredicateName);
    final EdmProperty edmProperty = keyPropertyRef == null ? null : keyPropertyRef.getProperty();
    if (edmProperty == null) {
      throw new UriValidationException(keyPredicateName + " is not a valid key property name.",
          UriValidationException.MessageKeys.INVALID_KEY_PROPERTY, keyPredicateName);
    }
    ParserHelper.requireNext(tokenizer, TokenKind.EQ);
    if (tokenizer.next(TokenKind.COMMA) || tokenizer.next(TokenKind.CLOSE) || tokenizer.next(TokenKind.EOF)) {
      throw new UriParserSyntaxException("Key value expected.", UriParserSyntaxException.MessageKeys.SYNTAX);
    }
    if (nextPrimitiveTypeValue(tokenizer, (EdmPrimitiveType) edmProperty.getType(), 
    		edmProperty.isNullable(), protocolType)) {
    	if (protocolType != null && protocolType.equalsIgnoreCase(REST)) {
    		if (odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.String).
    				equals((EdmPrimitiveType) edmProperty.getType())) {
    			return createUriParameter(edmProperty, keyPredicateName, "'"+tokenizer.getText()+"'", edm, 
    					referringType, aliases);
    		} else {
    			return createUriParameter(edmProperty, keyPredicateName, tokenizer.getText(), 
    					edm, referringType, aliases);
    		}
    	} else {
    		return createUriParameter(edmProperty, keyPredicateName, tokenizer.getText(), 
    				edm, referringType, aliases);
    	}
    } else {
    	if((protocolType != null && protocolType.equalsIgnoreCase(REST)) && 
    		(odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.String).
				equals((EdmPrimitiveType) edmProperty.getType()))) {
    		ParserHelper.nextPrimitiveValue(tokenizer);
    		return createUriParameter(edmProperty, keyPredicateName, "'"+tokenizer.getText()+"'", edm, 
					referringType, aliases);
    	}
      throw new UriParserSemanticException(keyPredicateName + " has not a valid  key value.",
          UriParserSemanticException.MessageKeys.INVALID_KEY_VALUE, keyPredicateName);
    }
  }

  protected static UriParameter createUriParameter(final EdmProperty edmProperty, final String parameterName,
      final String literalValue, final Edm edm, final EdmType referringType,
      final Map<String, AliasQueryOption> aliases) throws UriParserException, UriValidationException {
    final AliasQueryOption alias = literalValue.startsWith("@") ?
        getKeyAlias(literalValue, edmProperty, edm, referringType, aliases) :
        null;
    final String value = alias == null ? literalValue : alias.getText();
    final EdmPrimitiveType primitiveType = (EdmPrimitiveType) edmProperty.getType();
    try {
      if (!(primitiveType.validate(primitiveType.fromUriLiteral(value), edmProperty.isNullable(),
          edmProperty.getMaxLength(), edmProperty.getPrecision(), edmProperty.getScale(), edmProperty.isUnicode()))) {
        throw new UriValidationException("Invalid key property",
            UriValidationException.MessageKeys.INVALID_KEY_PROPERTY, parameterName);
      }
    } catch (final EdmPrimitiveTypeException e) {
      throw new UriValidationException("Invalid key property", e,
          UriValidationException.MessageKeys.INVALID_KEY_PROPERTY, parameterName);
    }

    return new UriParameterImpl()
        .setName(parameterName)
        .setText("null".equals(literalValue) ? null : literalValue)
        .setAlias(alias == null ? null : literalValue)
        .setExpression(alias == null ? null :
            alias.getValue() == null ? new LiteralImpl(value, primitiveType) : alias.getValue());
  }

  private static AliasQueryOption getKeyAlias(final String name, final EdmProperty edmProperty,
      final Edm edm, final EdmType referringType, final Map<String, AliasQueryOption> aliases)
      throws UriParserException, UriValidationException {
    if (aliases.containsKey(name)) {
      return parseAliasValue(name,
          edmProperty.getType(), edmProperty.isNullable(), edmProperty.isCollection(),
          edm, referringType, aliases);
    } else {
      throw new UriValidationException("Alias '" + name + "' for key value not found.",
          UriValidationException.MessageKeys.MISSING_ALIAS, name);
    }
  }

  protected static boolean nextPrimitiveTypeValue(UriTokenizer tokenizer,
      final EdmPrimitiveType primitiveType, final boolean nullable, String protocolType) {
    final EdmPrimitiveType type = primitiveType instanceof EdmTypeDefinition ?
        ((EdmTypeDefinition) primitiveType).getUnderlyingType() :
        primitiveType;
    if (tokenizer.next(TokenKind.ParameterAliasName)) {
      return true;
    } else if (nullable && tokenizer.next(TokenKind.NULL)) {
      return true;

    // Special handling for frequently-used types and types with more than one token kind.
    } else if (odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.Boolean).equals(type)) {
      return tokenizer.next(TokenKind.BooleanValue);
    } else if (odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.String).equals(type)) {
    	if (protocolType != null && protocolType.equalsIgnoreCase(REST)) {
    		return tokenizer.next(TokenKind.ODataIdentifier);	
    	} else {
    		return tokenizer.next(TokenKind.StringValue);
    	}
      
    } else if (odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.SByte).equals(type)
        || odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.Byte).equals(type)
        || odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.Int16).equals(type)
        || odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.Int32).equals(type)
        || odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.Int64).equals(type)) {
      return tokenizer.next(TokenKind.IntegerValue);
    } else if (odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.Guid).equals(type)) {
      return tokenizer.next(TokenKind.GuidValue);
    } else if (odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.Decimal).equals(type)) {
      // The order is important.
      // A decimal value should not be parsed as integer and let the tokenizer stop at the decimal point.
      return tokenizer.next(TokenKind.DecimalValue)
          || tokenizer.next(TokenKind.IntegerValue);
    } else if (odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.Double).equals(type)
        || odata.createPrimitiveTypeInstance(EdmPrimitiveTypeKind.Single).equals(type)) {
      // The order is important.
      // A floating-point value should not be parsed as decimal and let the tokenizer stop at 'E'.
      // A decimal value should not be parsed as integer and let the tokenizer stop at the decimal point.
      return tokenizer.next(TokenKind.DoubleValue)
          || tokenizer.next(TokenKind.DecimalValue)
          || tokenizer.next(TokenKind.IntegerValue);
    } else if (type.getKind() == EdmTypeKind.ENUM) {
      return tokenizer.next(TokenKind.EnumValue);
    } else {
      // Check the types that have not been checked already above.
      for (final Entry<TokenKind, EdmPrimitiveTypeKind> entry : tokenToPrimitiveType.entrySet()) {
        final EdmPrimitiveTypeKind kind = entry.getValue();
        if ((kind == EdmPrimitiveTypeKind.Date || kind == EdmPrimitiveTypeKind.DateTimeOffset
            || kind == EdmPrimitiveTypeKind.TimeOfDay || kind == EdmPrimitiveTypeKind.Duration
            || kind == EdmPrimitiveTypeKind.Binary
            || kind.isGeospatial())
            && odata.createPrimitiveTypeInstance(kind).equals(type)) {
          return tokenizer.next(entry.getKey());
        }
      }
      return false;
    }
  }

  protected static List<String> getParameterNames(final List<UriParameter> parameters) {
    List<String> names = new ArrayList<>();
    for (final UriParameter parameter : parameters) {
      names.add(parameter.getName());
    }
    return names;
  }

  protected static EdmStructuredType parseTypeCast(UriTokenizer tokenizer, final Edm edm,
      final EdmStructuredType referencedType) throws UriParserException {
    if (tokenizer.next(TokenKind.QualifiedName)) {
      final FullQualifiedName qualifiedName = new FullQualifiedName(tokenizer.getText());
      final EdmStructuredType type = referencedType.getKind() == EdmTypeKind.ENTITY ?
          edm.getEntityType(qualifiedName) :
          edm.getComplexType(qualifiedName);
      if (type == null) {
        throw new UriParserSemanticException("Type '" + qualifiedName + "' not found.",
            UriParserSemanticException.MessageKeys.UNKNOWN_PART, qualifiedName.getFullQualifiedNameAsString());
      } else {
        if (!type.compatibleTo(referencedType)) {
          throw new UriParserSemanticException("The type cast '" + qualifiedName + "' is not compatible.",
              UriParserSemanticException.MessageKeys.INCOMPATIBLE_TYPE_FILTER, type.getName());
        }
      }
      return type;
    }
    return null;
  }

  protected static EdmType getTypeInformation(final UriResourcePartTyped resourcePart) {
    EdmType type = null;
    if (resourcePart instanceof UriResourceWithKeysImpl) {
      final UriResourceWithKeysImpl lastPartWithKeys = (UriResourceWithKeysImpl) resourcePart;
      if (lastPartWithKeys.getTypeFilterOnEntry() != null) {
        type = lastPartWithKeys.getTypeFilterOnEntry();
      } else if (lastPartWithKeys.getTypeFilterOnCollection() != null) {
        type = lastPartWithKeys.getTypeFilterOnCollection();
      } else {
        type = lastPartWithKeys.getType();
      }

    } else if (resourcePart instanceof UriResourceTypedImpl) {
      final UriResourceTypedImpl lastPartTyped = (UriResourceTypedImpl) resourcePart;
      type = lastPartTyped.getTypeFilter() == null ?
          lastPartTyped.getType() :
          lastPartTyped.getTypeFilter();
    } else {
      type = resourcePart.getType();
    }

    return type;
  }

  protected static int parseNonNegativeInteger(final String optionName, final String optionValue,
      final boolean zeroAllowed) throws UriParserException {
    int value;
    try {
      value = Integer.parseInt(optionValue);
    } catch (final NumberFormatException e) {
      throw new UriParserSyntaxException("Illegal value of '" + optionName + "' option!", e,
          UriParserSyntaxException.MessageKeys.WRONG_VALUE_FOR_SYSTEM_QUERY_OPTION,
          optionName, optionValue);
    }
    if (value > 0 || value == 0 && zeroAllowed) {
      return value;
    } else {
      throw new UriParserSyntaxException("Illegal value of '" + optionName + "' option!",
          UriParserSyntaxException.MessageKeys.WRONG_VALUE_FOR_SYSTEM_QUERY_OPTION,
          optionName, optionValue);
    }
  }
  
  protected static void validateFunctionParameterFacets(final EdmFunction function, 
      final List<UriParameter> parameters, Edm edm, Map<String, AliasQueryOption> aliases) 
          throws UriParserException, UriValidationException {
    for (UriParameter parameter : parameters) {
      EdmParameter edmParameter = function.getParameter(parameter.getName());
      final EdmType type = edmParameter.getType();
      final EdmTypeKind kind = type.getKind();
      if ((kind == EdmTypeKind.PRIMITIVE || kind == EdmTypeKind.DEFINITION || kind == EdmTypeKind.ENUM)
          && !edmParameter.isCollection()) {
        final EdmPrimitiveType primitiveType = (EdmPrimitiveType) type;
        String text = null;
        try {
          text = parameter.getAlias() == null ?
              parameter.getText() :
                aliases.containsKey(parameter.getAlias()) ?
                    parseAliasValue(parameter.getAlias(),
                        edmParameter.getType(), edmParameter.isNullable(), edmParameter.isCollection(),
                        edm, type, aliases).getText() : null;
          if (edmParameter.getMapping() == null) {
            primitiveType.valueOfString(primitiveType.fromUriLiteral(text),
                edmParameter.isNullable(), edmParameter.getMaxLength(), edmParameter.getPrecision(), 
                edmParameter.getScale(), true, primitiveType.getDefaultType());
          } else {
            primitiveType.valueOfString(primitiveType.fromUriLiteral(text),
                edmParameter.isNullable(), edmParameter.getMaxLength(), edmParameter.getPrecision(), 
                edmParameter.getScale(), true, edmParameter.getMapping().getMappedJavaClass());
          }
        } catch (final EdmPrimitiveTypeException e) {
          throw new UriValidationException(
              "Invalid value '" + text + "' for parameter " + parameter.getName(), e,
              UriValidationException.MessageKeys.INVALID_VALUE_FOR_PROPERTY, parameter.getName());
        }
      }
    }
  }

  public static boolean checkSameResource(UriResource prev, UriResource actual) {
    if (prev.getSegmentValue().equals(actual.getSegmentValue()) && prev.getKind() == actual.getKind()) {
      if (prev.getKind() == UriResourceKind.entitySet) {
        return (((UriResourceEntitySet) actual).getKeyPredicates().size() >= ((UriResourceEntitySet) prev)
            .getKeyPredicates().size());
      }
      if (prev.getKind() == UriResourceKind.navigationProperty) {
        return (((UriResourceNavigation) actual).getKeyPredicates().size() >= ((UriResourceNavigation) prev)
            .getKeyPredicates().size());
      }
    }
    return false;
  }

  public static void checkNavigationKeys(List<UriResource> l) throws UriParserSemanticException {
    int size = l.size();
    for (int x = 0; x < size; x++) {
      UriResource res = l.get(x);
      UriResource nav = x == size - 1 ? null : l.get(x + 1);
      if (res.getKind() == UriResourceKind.entitySet) {
        UriResourceEntitySet es = (UriResourceEntitySet) res;
        boolean hasNavigation = nav != null && nav.getKind() == UriResourceKind.navigationProperty;
        int keySize = es.getEntityType().getKeyPropertyRefs().size();
        int keyPreds = es.getKeyPredicates().size();
        if (!keysMatch(keySize, keyPreds, hasNavigation)) {
          throw new UriParserSemanticException("Wrong number of key properties",
              UriParserSemanticException.MessageKeys.WRONG_NUMBER_OF_KEY_PROPERTIES,
              Integer.toString(keySize), Integer.toString(keyPreds));
        }
      }
    }
  }

  private static boolean keysMatch(int keySize, int keyPreds, boolean hasNavigation) {
    if (!hasNavigation) {
      return (keyPreds == keySize || keyPreds == 0);
    }
    return (keyPreds == keySize);
  }
}
