/*
 * Decompiled with CFR 0.152.
 */
package com.sap.cloud.adk.checks;

import com.sap.cloud.adk.build.BuildProvider;
import com.sap.cloud.adk.checks.BlueprintMetadataValidator;
import com.sap.cloud.adk.checks.ComponentMetadataValidator;
import com.sap.cloud.adk.checks.ComponentStructureValidator;
import com.sap.cloud.adk.checks.VersionComparator;
import com.sap.cloud.adk.exception.CheckException;
import com.sap.cloud.adk.util.AdapterMetadataBuildUtil;
import com.sap.cloud.adk.util.BuildProviderUtil;
import com.sap.cloud.adk.util.FileUtil;
import com.sap.cloud.adk.util.MarkerInfo;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPathExpressionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class AdapterProjectChecks {
    private static final Logger log = LoggerFactory.getLogger(AdapterProjectChecks.class);
    private static final String BP_METADATA_XML = "bpMetadata.xml";
    private static final String LIBS = "libs";
    private static final String TARGET = "target";
    private static final String META_INF = "META-INF";
    private static final String INVALID_CONTENT = "Invalid content. Please check the metadata folder for metadata or bpMetadata files.";
    private static final String REFERENCE_NAME_XPATH = "//AttributeReference/ReferenceName";
    private static final String PATH_TO_ENDPOINT_CLASS = "META-INF/services/org/apache/camel/component/";
    private static final String EMPTY_DIRECTORY = "No files in the directory: ";
    private boolean newComponentProject = false;

    public AdapterProjectChecks(boolean isNewComponentProject) {
        this.newComponentProject = isNewComponentProject;
    }

    private static boolean checkBlueprintMetadataExists(String bmdPath) {
        File blueprintMetadataFile = new File(bmdPath);
        return blueprintMetadataFile.exists();
    }

    private static boolean doesCMDFileExist(List<String> allCMDPath) {
        return !allCMDPath.isEmpty();
    }

    private static ComponentMetadataValidator validateMetadata(List<MarkerInfo> infoList, String cmdPath) throws CheckException, SAXException, IOException, ParserConfigurationException, XPathExpressionException {
        ComponentMetadataValidator cmdValidator = new ComponentMetadataValidator(new File(cmdPath));
        cmdValidator.validateComponentMetadata(infoList);
        return cmdValidator;
    }

    private static void validateBlueprintMetadata(List<MarkerInfo> infoList, String bmdPath, NodeList cmdAttributeNodeList) throws XPathExpressionException, CheckException, SAXException, IOException, ParserConfigurationException {
        BlueprintMetadataValidator bpMetadataValidator = new BlueprintMetadataValidator(new File(bmdPath), cmdAttributeNodeList);
        bpMetadataValidator.validateBlueprintMetadata(infoList);
    }

    private String getBMDPath(File sourceProject) {
        Properties manifestProperties = BuildProviderUtil.getManifestProperties(sourceProject);
        String directoryPath = BuildProviderUtil.isNewComponentProject(manifestProperties) ? AdapterMetadataBuildUtil.getCMDDirectoryPathForProject(sourceProject) : AdapterMetadataBuildUtil.getCMDDirectoryPathForReuseComponentProject(sourceProject);
        return directoryPath + File.separatorChar + BP_METADATA_XML;
    }

    private String getLibDirectoryPath(File sourceProject) {
        String basePath = sourceProject.getAbsolutePath();
        if (this.newComponentProject) {
            return basePath + File.separatorChar + TARGET + File.separatorChar + LIBS;
        }
        return basePath + File.separatorChar + LIBS;
    }

    private String getComponentDirectoryPath(File sourceProject) {
        String basePath = sourceProject.getAbsolutePath();
        if (this.newComponentProject) {
            return basePath + File.separatorChar + TARGET;
        }
        return basePath + File.separatorChar + "component";
    }

    public void checkAdapterProject(File sourceProject, List<MarkerInfo> infoList) throws CheckException {
        boolean deleteFileWithChildren;
        if (sourceProject == null) {
            throw new CheckException("Invalid project");
        }
        ComponentMetadataValidator cmdValidator = null;
        String srcMetadata = BuildProvider.getConfigFilePath(sourceProject);
        String manifestFile = sourceProject.getAbsolutePath() + File.separatorChar + META_INF + File.separatorChar;
        List<String> allCMDPath = AdapterMetadataBuildUtil.getAllCMDFilesInProject(sourceProject);
        String bmdPath = this.getBMDPath(sourceProject);
        if (!new File(srcMetadata).exists()) {
            String msg = "Invalid content. Please check that the project contains a config file";
            MarkerInfo info = new MarkerInfo(msg, -1, null, 2);
            infoList.add(info);
            throw new CheckException(msg);
        }
        File manifestFolder = new File(manifestFile);
        if (manifestFolder.exists() && !(deleteFileWithChildren = FileUtil.deleteFileWithChildren(manifestFolder))) {
            MarkerInfo info = new MarkerInfo(manifestFolder.getName() + " could not be deleted", -1, null, 1);
            infoList.add(info);
        }
        this.checkProjectDirectoryStructure(sourceProject, infoList);
        try {
            this.checkForDuplicateMetadataVariants(infoList, allCMDPath);
            for (String cmdPath : allCMDPath) {
                cmdValidator = AdapterProjectChecks.validateMetadata(infoList, cmdPath);
                cmdValidator.checkConsistencyCMDAndManifest(srcMetadata, infoList);
                this.checkForEndPointNameConsistency(sourceProject, cmdValidator, infoList);
            }
            if (cmdValidator != null && AdapterProjectChecks.checkBlueprintMetadataExists(bmdPath)) {
                AdapterProjectChecks.validateBlueprintMetadata(infoList, bmdPath, cmdValidator.getNodeList(REFERENCE_NAME_XPATH));
                this.validateReferencedComponentsForBlueprintMetadata(infoList, bmdPath, allCMDPath);
            }
        }
        catch (Exception e) {
            String exceptionMessage = e.getMessage();
            throw new CheckException(exceptionMessage, e);
        }
        for (String cmdPath : allCMDPath) {
            File metadataFile = new File(cmdPath);
            this.componentStructureValidations(sourceProject, metadataFile, infoList);
        }
    }

    private void componentStructureValidations(File sourceProject, File metadataFile, List<MarkerInfo> infoList) throws CheckException {
        ComponentStructureValidator cmdStructureValidator = new ComponentStructureValidator(this.newComponentProject);
        try {
            cmdStructureValidator.validatePollingConsumerRequirements(sourceProject, metadataFile, infoList);
        }
        catch (Exception e) {
            String exceptionMessage = e.getMessage();
            throw new CheckException(exceptionMessage, e);
        }
    }

    private void checkForEndPointNameConsistency(File sourceProject, ComponentMetadataValidator cmdValidator, List<MarkerInfo> infoList) throws CheckException, IOException {
        HashSet<String> endPointNames = new HashSet<String>();
        if (this.newComponentProject) {
            String componentDirectoryPath = AdapterMetadataBuildUtil.getComponentFilePath(sourceProject);
            endPointNames.add(this.getFileName(componentDirectoryPath));
        } else {
            String jarFileLocation = this.getJarFileLocation(sourceProject);
            String componentJarFileName = this.getFileName(jarFileLocation);
            if (componentJarFileName.endsWith(".jar")) {
                try (ZipInputStream zip = new ZipInputStream(new FileInputStream(jarFileLocation + File.separatorChar + componentJarFileName));){
                    ZipEntry entry = zip.getNextEntry();
                    while (entry != null) {
                        if (!entry.isDirectory() && entry.getName().contains(PATH_TO_ENDPOINT_CLASS)) {
                            int lastIndex = entry.getName().lastIndexOf("/");
                            endPointNames.add(entry.getName().substring(lastIndex + 1));
                        }
                        entry = zip.getNextEntry();
                    }
                }
            } else {
                endPointNames.add(componentJarFileName);
            }
        }
        cmdValidator.compareEndPointNameAndUpdateInfoList(infoList, endPointNames);
    }

    private String getJarFileLocation(File sourceProject) {
        return sourceProject.getAbsolutePath() + File.separatorChar + "component";
    }

    private String getFileName(String directoryLocation) throws CheckException {
        try {
            File file = new File(directoryLocation);
            File[] files = file.listFiles();
            if (files == null) {
                throw new CheckException(EMPTY_DIRECTORY + directoryLocation);
            }
            for (File fileEntry : files) {
                if (fileEntry.isDirectory()) continue;
                return fileEntry.getName();
            }
        }
        catch (Exception ex) {
            log.error("Exception occurred while fetching end point file name from location {}", (Object)directoryLocation, (Object)ex);
            throw new CheckException("Exception occurred while fetching end point file name");
        }
        return "sap-sample";
    }

    private void checkForDuplicateMetadataVariants(List<MarkerInfo> infoList, List<String> allCMDPath) throws CheckException {
        ArrayList<String> allVariantsOfAdapter = new ArrayList<String>();
        for (String cmdPath : allCMDPath) {
            try {
                List<String> variantsInCMD = AdapterMetadataBuildUtil.getVariantIds(AdapterMetadataBuildUtil.bytesToXml(AdapterMetadataBuildUtil.getMetadataInputStreamFromPath(cmdPath)));
                for (String variant : variantsInCMD) {
                    String variantWithVersion = AdapterMetadataBuildUtil.getVariantWithVersion(variant);
                    if (allVariantsOfAdapter.contains(variantWithVersion)) {
                        String errorMessage = "Invalid Content: Duplicate Variant - " + variant + "  in " + cmdPath;
                        MarkerInfo info = new MarkerInfo(errorMessage, 1, cmdPath, 2);
                        infoList.add(info);
                        log.error(errorMessage);
                        throw new CheckException(errorMessage);
                    }
                    allVariantsOfAdapter.add(variantWithVersion);
                }
            }
            catch (IOException | ParserConfigurationException e) {
                log.error("Cannot Parse the CMD {}", (Object)cmdPath);
                throw new CheckException("Cannot Parse the CMD " + cmdPath + ":" + e.getMessage());
            }
        }
        this.checkVariantsDistribution(allVariantsOfAdapter, infoList);
    }

    private void checkVariantsDistribution(List<String> allVariantsList, List<MarkerInfo> infoList) throws CheckException {
        MarkerInfo info;
        String errorMessage;
        if (allVariantsList.isEmpty()) {
            return;
        }
        HashMap<String, List<String>> sendersSignificantVersions = new HashMap<String, List<String>>();
        HashMap<String, List<String>> receiversSignificantVersions = new HashMap<String, List<String>>();
        for (String variantId : allVariantsList) {
            String variantMainId = AdapterMetadataBuildUtil.getVarinatIdWithoutVersion(variantId);
            String significantVersion = VersionComparator.removeMicroVersion(AdapterMetadataBuildUtil.getVariantVersion(variantId));
            List versionsList = null;
            if (AdapterMetadataBuildUtil.isSender(variantId)) {
                sendersSignificantVersions.putIfAbsent(variantMainId, new ArrayList());
                versionsList = (List)sendersSignificantVersions.get(variantMainId);
            } else {
                receiversSignificantVersions.putIfAbsent(variantMainId, new ArrayList());
                versionsList = (List)receiversSignificantVersions.get(variantMainId);
            }
            versionsList.add(significantVersion);
        }
        if (!sendersSignificantVersions.isEmpty() && !this.checkIfItemsHaveDifferentSignificantVersion(sendersSignificantVersions)) {
            errorMessage = "Invalid Versions: All Senders should have different significant version (X.X. Micro version ignored)";
            info = new MarkerInfo(errorMessage, 1, "", 2);
            infoList.add(info);
            log.error(errorMessage);
            throw new CheckException(errorMessage);
        }
        if (!receiversSignificantVersions.isEmpty() && !this.checkIfItemsHaveDifferentSignificantVersion(receiversSignificantVersions)) {
            errorMessage = "Invalid Versions: All Receivers should have different significant version (X.X. Micro version ignored)";
            info = new MarkerInfo(errorMessage, 1, "", 2);
            infoList.add(info);
            log.error(errorMessage);
            throw new CheckException(errorMessage);
        }
    }

    private boolean checkIfItemsHaveDifferentSignificantVersion(Map<String, List<String>> significantVersionsMap) {
        if (significantVersionsMap.isEmpty()) {
            return true;
        }
        for (Map.Entry<String, List<String>> entry : significantVersionsMap.entrySet()) {
            if (this.checkIfItemsHaveDifferentSignificantVersion(entry.getValue())) continue;
            return false;
        }
        return true;
    }

    private boolean checkIfItemsHaveDifferentSignificantVersion(List<String> significantVersionsList) {
        if (significantVersionsList.isEmpty() || significantVersionsList.size() == 1) {
            return true;
        }
        String[] versions = significantVersionsList.toArray(new String[significantVersionsList.size()]);
        for (int i = 0; i < significantVersionsList.size() - 1; ++i) {
            for (int j = i + 1; j < significantVersionsList.size(); ++j) {
                if (!versions[i].equals(versions[j])) continue;
                return false;
            }
        }
        return true;
    }

    private void checkIfDirectoryContainsOnlyJars(File directory) throws CheckException {
        File[] files = directory.listFiles();
        if (files == null) {
            throw new CheckException(EMPTY_DIRECTORY + directory.getName());
        }
        for (File jarFiles : files) {
            if (jarFiles.getName().endsWith("jar")) continue;
            throw new CheckException("Invalid content. Please check that the " + directory.getName() + " folder contains only *.jar files.");
        }
    }

    private void checkIfDirectoryContainsOnlyCMD(File directory, List<MarkerInfo> infoList) throws CheckException {
        File[] files = directory.listFiles();
        if (files == null) {
            throw new CheckException(EMPTY_DIRECTORY + directory.getName());
        }
        for (File file : files) {
            if (file.getName().toLowerCase().contains("metadata") || file.getName().toLowerCase().contains("bpmetadata")) continue;
            MarkerInfo info = new MarkerInfo(INVALID_CONTENT, -1, null, 2);
            infoList.add(info);
            throw new CheckException(INVALID_CONTENT);
        }
    }

    private void checkProjectDirectoryStructure(File sourceProject, List<MarkerInfo> infoList) throws CheckException {
        boolean metadataFound;
        File libDirectory = new File(this.getLibDirectoryPath(sourceProject));
        if (libDirectory.exists()) {
            this.checkIfDirectoryContainsOnlyJars(libDirectory);
        }
        File componentDirectory = new File(this.getComponentDirectoryPath(sourceProject));
        if (!this.newComponentProject) {
            this.checkIfDirectoryContainsOnlyJars(componentDirectory);
        }
        if (!(metadataFound = AdapterProjectChecks.doesCMDFileExist(AdapterMetadataBuildUtil.getAllCMDFilesInProject(sourceProject)))) {
            String msg = "Invalid content. Please check that the metadata folder contains a metadata.xml file.";
            MarkerInfo info = new MarkerInfo(msg, -1, null, 2);
            infoList.add(info);
            throw new CheckException(msg);
        }
        Properties manifestProperties = BuildProviderUtil.getManifestProperties(sourceProject);
        String directoryPath = BuildProviderUtil.isNewComponentProject(manifestProperties) ? AdapterMetadataBuildUtil.getCMDDirectoryPathForProject(sourceProject) : AdapterMetadataBuildUtil.getCMDDirectoryPathForReuseComponentProject(sourceProject);
        this.checkIfDirectoryContainsOnlyCMD(new File(directoryPath), infoList);
    }

    private void validateReferencedComponentsForBlueprintMetadata(List<MarkerInfo> infoList, String bmdPath, List<String> allCMDPath) throws CheckException, SAXException, IOException, ParserConfigurationException {
        File bpMetadataFile = new File(bmdPath);
        List<String> bpMetadataVariantIds = AdapterMetadataBuildUtil.getVariantIds(AdapterMetadataBuildUtil.getDocumentFromFile(bpMetadataFile));
        ArrayList<String> referencedComponents = new ArrayList<String>();
        for (String cmdPath : allCMDPath) {
            File metadataFile = new File(cmdPath);
            referencedComponents.addAll(AdapterMetadataBuildUtil.getReferencedComponents(metadataFile));
        }
        if (!this.isReferencedComponentPresentInMetadata(bpMetadataVariantIds, referencedComponents)) {
            String errorMsg = "Blueprint Metadata is not referred in the Component Metadata";
            MarkerInfo info = new MarkerInfo(errorMsg, 2, bmdPath, 2);
            infoList.add(info);
            throw new CheckException("Validation Failed:  " + errorMsg);
        }
    }

    private boolean isReferencedComponentPresentInMetadata(List<String> bpMetadataVariantIds, List<String> referencedComponents) {
        List<String> microToMinorBpVariantId = VersionComparator.convertMicroToMinorVersionFromListOfComponentId(bpMetadataVariantIds);
        List<String> microToMinorRefVariantId = VersionComparator.convertMicroToMinorVersionFromListOfComponentId(referencedComponents);
        return this.isContainedInList(microToMinorBpVariantId, microToMinorRefVariantId);
    }

    private boolean isContainedInList(List<String> firstList, List<String> secondList) {
        HashSet<String> firstSet = new HashSet<String>(firstList);
        HashSet<String> secondSet = new HashSet<String>(secondList);
        return secondSet.containsAll(firstSet);
    }
}

