/*
 * ----------------------------------------------------------------
 * © 2019-2021 SAP SE or an SAP affiliate company. All rights reserved.
 * ----------------------------------------------------------------
 *
 */
package com.sap.cds.mtx.impl;

import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.github.benmanes.caffeine.cache.Ticker;
import com.sap.cds.CdsException;
import com.sap.cds.mtx.MetaDataAccessor;
import com.sap.cds.mtx.ModelId;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.impl.CdsModelReader;

/**
 * Class that provides access to CDS and Edmx models and caches them
 *
 * @param <M> Type used for the Edmx model
 */
public class MetaDataAccessorImpl<M> implements MetaDataAccessor<M> {

	private static final long NANOS_TO_SECONDS = 1000 * 1000 * 1000l;
	private static final Logger logger = LoggerFactory.getLogger(MetaDataAccessorImpl.class);
	private final Cache<CdsModel> modelIdToCdsModel;
	private final Cache<M> modelIdToEdmxModel;

	@FunctionalInterface
	public static interface EdmxModelCreator<M> {
		M parse(String edmx, String serviceName);
	}

	/**
	 * @param sidecarAccess Object of type {@link SidecarAccess}that provides access
	 *                      to the node.js application sidecar/mtx via a rest API
	 * @param cacheParams   Parameters that control size and lifecycle of cache for
	 *                      cds and edmx models
	 * @param strToEdmx     Function that converts an edmx model description given
	 *                      as string into an edmx model
	 * @param cacheTicker   Optional ticker used by guava cache for testing
	 *                      purposes, use null for productive use
	 */
	public MetaDataAccessorImpl(SidecarAccess sidecarAccess, CacheParams cacheParams, EdmxModelCreator<M> strToEdmx,
			Ticker cacheTicker) {
		if (cacheTicker == null) {
			cacheTicker = Ticker.systemTicker();
		}

		ExecutorService executorService = Executors.newSingleThreadExecutor();
		modelIdToCdsModel = new CdsCache(sidecarAccess, cacheParams, cacheTicker, executorService);
		modelIdToEdmxModel = new EdmxCache<M>(sidecarAccess, strToEdmx, cacheParams, cacheTicker, executorService);
	}

	@Override
	public CdsModel getCdsModel(ModelId key, int maxAgeSeconds) {
		return modelIdToCdsModel.getOrLoadIfStale(key, maxAgeSeconds);
	}

	@Override
	public M getEdmx(ModelId key, int maxAgeSeconds) throws CdsException {
		return modelIdToEdmxModel.getOrLoadIfStale(key, maxAgeSeconds);
	}

	@Override
	public void evict(String tenantId) {
		modelIdToCdsModel.evict(tenantId);
		modelIdToEdmxModel.evict(tenantId);
		// inform listeners
	}

	@Override
	public void refresh(String tenantId, int maxAgeSeconds) {
		modelIdToCdsModel.refresh(tenantId, maxAgeSeconds);
		modelIdToEdmxModel.refresh(tenantId, maxAgeSeconds);
		// inform listeners
	}

	private static abstract class Cache<V> {
		private final Ticker ticker;
		private final LoadingCache<ModelId, Entry<V>> cache;
		private final String cacheName = getClass().getSimpleName();

		protected Cache(CacheParams params, Ticker ticker, ExecutorService executorService) {
			this.ticker = ticker;
			this.cache = Caffeine.newBuilder().maximumSize(params.getMaximumSize())
					.expireAfterAccess(params.getExpirationDuration(), params.getExpirationDurationUnit())
					.refreshAfterWrite(params.getRefreshDuration(), params.getRefreshDurationUnit())
					.executor(executorService)
					.ticker(ticker)
					.evictionListener((k, v, c) -> {
						if (c.wasEvicted()) {
							logger.debug("Evicted '{}' in cache '{}' with cause '{}'", k, cacheName, c);
						}
					})
					.build(new CacheLoader<ModelId, Entry<V>>() {
						@Override
						public Entry<V> load(ModelId key) {
							return Cache.this.load(key, null);
						}

						@Override
						public Entry<V> reload(ModelId key, Entry<V> oldValue) {
							logger.debug("Reloading '{}' in cache '{}'", key, cacheName);
							try {
								return Cache.this.load(key, oldValue);
							} catch (Exception e) {// NOSONAR
								logger.error("Reloading '{}' failed", key, e);
								return oldValue;
							}
						}
					});
		}

		public void evict(String tenantId) {
			logger.debug("Evicting tenant '{}' from cache '{}'", tenantId, cacheName);
			forTenant(tenantId, cache::invalidate);
		}

		public void refresh(String tenantId, int maxAgeSeconds) {
			logger.debug("Refreshing tenant '{}' in cache '{}'", tenantId, cacheName);
			forTenant(tenantId, k -> getOrLoadIfStale(k, maxAgeSeconds));
		}

		private void forTenant(String tenantId, Consumer<ModelId> action) {
			cache.asMap().keySet().stream().filter(k -> Objects.equals(tenantId, k.getTenantId())).forEach(action);
		}

		public V getOrLoadIfStale(ModelId key, int maxAgeSeconds) {
			long maxAgeNanos = maxAgeSeconds * NANOS_TO_SECONDS;
			Entry<V> entry;
			try {
				entry = cache.get(key);
			} catch (RuntimeException e) {
				throw new CdsException(e);
			}
			if ((ticker.read() - entry.refreshed()) > maxAgeNanos) {
				// sync load
				Entry<V> loaded = load(key, entry);
				if (loaded != entry) {
					cache.put(key, loaded);
					entry = loaded;
				}
			} else {
				logger.debug("'{}' in cache '{}' is not older than '{}'", key, cacheName, maxAgeSeconds);
			}

			return entry.getEntry();
		}

		private Entry<V> load(ModelId key, Entry<V> oldEntry) {
			logger.debug("Loading '{}' in cache '{}'", key, cacheName);
			String eTag = oldEntry != null ? oldEntry.getETag() : null;
			long beforeAccess = ticker.read();
			ModelAndInformation model = access(key, eTag);
			if (oldEntry != null && model.isNotModified()) {
				oldEntry.refresh(beforeAccess);
				logger.debug("Refreshed unchanged '{}' in cache '{}'", key, cacheName);
				return oldEntry;
			}

			// model has changed -> notify
			return new Entry<V>(parse(key, model.getModel()), model.getETag(), beforeAccess);
		}

		abstract ModelAndInformation access(ModelId key, String eTag);

		abstract V parse(ModelId key, String model);

	}

	private static class Entry<V> {
		private final V entry;
		private final String eTag;
		private final AtomicLong refreshed;

		public Entry(V entry, String eTag, long refreshed) {
			this.entry = entry;
			this.eTag = eTag != null ? eTag.trim() : null;
			this.refreshed = new AtomicLong(refreshed);
		}

		public V getEntry() {
			return entry;
		}

		public String getETag() {
			return eTag;
		}

		public void refresh(long refreshed) {
			this.refreshed.set(refreshed);
		}

		public long refreshed() {
			return refreshed.get();
		}

	}

	private static class EdmxCache<M> extends Cache<M> {

		private final SidecarAccess sidecarAccess;
		private final EdmxModelCreator<M> strToEdmx;

		public EdmxCache(SidecarAccess sidecarAccess, EdmxModelCreator<M> strToEdmx, CacheParams params, Ticker ticker, ExecutorService executorService) {
			super(params, ticker, executorService);
			this.sidecarAccess = sidecarAccess;
			this.strToEdmx = strToEdmx;
		}

		@Override
		ModelAndInformation access(ModelId key, String eTag) {
			return sidecarAccess.getEdmx(key, eTag);
		}

		@Override
		M parse(ModelId key, String model) {
			return strToEdmx.parse(model, key.getServiceName().orElse(null));
		}

	}

	private static class CdsCache extends Cache<CdsModel> {

		private final SidecarAccess sidecarAccess;

		public CdsCache(SidecarAccess sidecarAccess, CacheParams params, Ticker ticker, ExecutorService executorService) {
			super(params, ticker, executorService);
			this.sidecarAccess = sidecarAccess;
		}

		@Override
		ModelAndInformation access(ModelId key, String eTag) {
			return sidecarAccess.getCsn(key, eTag);
		}

		@Override
		CdsModel parse(ModelId key, String csn) {
			return CdsModelReader.read(csn, true);
		}

	}

}
