/*
 * ----------------------------------------------------------------
 * © 2019-2021 SAP SE or an SAP affiliate company. All rights reserved.
 * ----------------------------------------------------------------
 *
 */

package com.sap.cds.mtx.impl;

import java.util.HashMap;
import java.util.Map;

import com.sap.cds.mtx.ModelId;
import com.sap.cloud.mt.tools.api.ResilienceConfig;
import com.sap.cloud.mt.tools.api.ServiceCall;
import com.sap.cloud.mt.tools.exception.InternalException;
/**
 * Class that is responsible for communication with node.js application
 * sidecar/mtx via its REST API
 */

public class SidecarAccessV2 extends AbstractSidecarAccess implements SidecarAccess {

	private static final String BASE_PATH = "/mtx/v2/model/";
	private static final String CSN_PATH = BASE_PATH + "csn/";
	private static final String EDMX_PATH = BASE_PATH + "edmx/";

	/**
	 * @param sidecarBaseUrl   URL of sidecar application without path specification
	 * @param authenticator    object that is responsible for authentication
	 * @param resilienceConfig Parameters like number of retries, wait between retries, ..
	 */
	public SidecarAccessV2(String sidecarBaseUrl, Authenticator authenticator, ResilienceConfig resilienceConfig) {
		super(sidecarBaseUrl, authenticator, resilienceConfig);
	}

	/**
	 * @param sidecarBaseUrl URL of sidecar application without path specification
	 * @param authenticator  object that is responsible for authentication
	 */
	@Deprecated
	public SidecarAccessV2(String sidecarBaseUrl, Authenticator authenticator) {
		this(sidecarBaseUrl, authenticator, ResilienceConfig.NONE);
	}

	@Override
	protected String getDestinationName() {
		return SidecarAccessV1.MTX_SIDECAR_DESTINATION;
	}

	@Override
	protected String getCsnPath() {
		return CSN_PATH;
	}

	@Override
	protected ServiceCall createCsnCall(ModelId id, String eTag) throws InternalException {
		return csnEndpoint.createServiceCall()
				.http()
				.post()
				.payload(getCsnBody(id))
				.noPathParameter()
				.noQuery()
				.authenticationTokenSupplier(getAuthenticationTokenSupplier())
				.insertHeaderFields(getHeaderFields(eTag))
				.end();
	}

	@Override
	protected String getEdmxPath() {
		return EDMX_PATH;
	}

	@Override
	protected ServiceCall createEdmxCall(ModelId id, String eTag) throws InternalException {
		return edmxEndpoint.createServiceCall()
				.http()
				.post()
				.payload(getEdmxBody(id))
				.noPathParameter()
				.noQuery()
				.authenticationTokenSupplier(getAuthenticationTokenSupplier())
				.insertHeaderFields(getHeaderFields(eTag))
				.end();
	}

	protected Map<String, Object> getBody(ModelId id) {
		Map<String, Object> body = new HashMap<>();
		body.put("tenant", id.getTenantId());
		body.put("toggles", id.getFeatures());

		return body;
	}

	protected Map<String, Object> getCsnBody(ModelId id) {
		return getBody(id);
	}

	protected Map<String, Object> getEdmxBody(ModelId id) {
		Map<String, Object> body = getBody(id);
		id.getServiceName().ifPresent(s -> body.put("service", s));
		id.getLanguage().ifPresent(l -> body.put("locale", l));
		id.getODataVersion().ifPresent(v -> body.put("odataVersion", v));

		return body;
	}
}
