/*
 * ----------------------------------------------------------------
 * © 2019-2021 SAP SE or an SAP affiliate company. All rights reserved.
 * ----------------------------------------------------------------
 *
 */

package com.sap.cds.mtx.impl;

import static java.nio.charset.StandardCharsets.UTF_8;

import java.util.Collections;
import java.util.Map;

import org.apache.http.HttpHeaders;

import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.sap.cds.mtx.ModelId;
import com.sap.cloud.mt.tools.api.ResilienceConfig;
import com.sap.cloud.mt.tools.api.ServiceResponse;
import com.sap.cloud.mt.tools.exception.InternalException;

/**
 * Class that is responsible for communication with external model provider
 * service
 */

public class SidecarAccessV3 extends SidecarAccessV2 {

	private static final HashFunction hasher = Hashing.goodFastHash(160);

	protected static final String BASE_PATH = "/-/cds/model-provider/";
	private static final String CSN_PATH = BASE_PATH + "getOdataCsn";
	private static final String EDMX_PATH = BASE_PATH + "getEdmx";

	private static final String MODEL_PROVIDER_DESTINATION = "com.sap.cds.modelProvider";

	/**
	 * @param sidecarBaseUrl   URL of sidecar application without path specification
	 * @param authenticator    object that is responsible for authentication
	 * @param resilienceConfig Parameters like number of retries, wait between
	 *                         retries, ..
	 */
	public SidecarAccessV3(String sidecarBaseUrl, Authenticator authenticator, ResilienceConfig resilienceConfig) {
		super(sidecarBaseUrl, authenticator, resilienceConfig);
	}

	@Deprecated
	public SidecarAccessV3(String sidecarBaseUrl, Authenticator authenticator) {
		super(sidecarBaseUrl, authenticator, ResilienceConfig.NONE);
	}

	@Override
	protected String getDestinationName() {
		return MODEL_PROVIDER_DESTINATION;
	}

	@Override
	protected Map<String, String> getHeaderFields(String eTag) {
		return Collections.singletonMap(HttpHeaders.ACCEPT, APPLICATION_JSON);
	}

	@Override
	protected String getCsnPath() {
		return CSN_PATH;
	}

	@Override
	protected ModelAndInformation csnModelInfo(ServiceResponse<String> response, String oldHash)
			throws InternalException {
		return modelInfo(getPayload(response), oldHash);
	}

	private String getPayload(ServiceResponse<String> response) throws InternalException {
		return response.getPayload().orElse("");
	}

	@Override
	protected String getEdmxPath() {
		return EDMX_PATH;
	}

	@Override
	protected ModelAndInformation edmxModelInfo(ServiceResponse<String> response, String oldHash)
			throws InternalException {
		return modelInfo(getPayload(response), oldHash);
	}

	private ModelAndInformation modelInfo(String value, String oldHash) {
		String newHash = hasher.hashString(value, UTF_8).toString();
		if (newHash.equals(oldHash)) {
			return new ModelAndInformation(null, oldHash, true);
		} else {
			return new ModelAndInformation(value, newHash, false);
		}
	}

	@Override
	protected Map<String, Object> getEdmxBody(ModelId id) {
		Map<String, Object> body = getBody(id);
		id.getServiceName().ifPresent(s -> body.put("serviceName", s));
		id.getLanguage().ifPresent(l -> body.put("locale", l));
		id.getODataVersion().ifPresent(v -> body.put("odataFlavor", v));

		return body;
	}

}
