/*
 * ----------------------------------------------------------------
 * © 2019-2021 SAP SE or an SAP affiliate company. All rights reserved.
 * ----------------------------------------------------------------
 *
 */
package com.sap.cds.mtx.impl;

import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.github.benmanes.caffeine.cache.Ticker;
import com.sap.cds.CdsException;
import com.sap.cds.mtx.MetaDataAccessor;
import com.sap.cds.mtx.ModelId;
import com.sap.cds.reflect.CdsModel;
import com.sap.cds.reflect.impl.CdsModelReader;

/**
 * Class that provides access to CDS and Edmx models and caches them
 *
 * @param <M> Type used for the Edmx model
 */
public class MetaDataAccessorImpl<M> implements MetaDataAccessor<M> {

	private static final long NANOS_TO_SECONDS = 1000 * 1000 * 1000L;
	private static final Logger logger = LoggerFactory.getLogger(MetaDataAccessorImpl.class);
	private final Cache<CdsModel> modelIdToCdsModel;
	private final Cache<M> modelIdToEdmxModel;

	@FunctionalInterface
	public interface EdmxModelCreator<M> {
		M parse(String edmx, String serviceName);
	}

	public static class MetaDataAccessorConfig {

		private SidecarAccess sidecarAccess;
		private CacheParams cacheParams;
		private EdmxModelCreator<?> strToEdmx;
		private Function<String, CdsModel> strToModel;

		private MetaDataAccessorConfig() {
		}

		public SidecarAccess getSidecarAccess() {
			return sidecarAccess;
		}

		public CacheParams getCacheParams() {
			return cacheParams;
		}

		public EdmxModelCreator<?> getStrToEdmx() {
			return strToEdmx;
		}

		public Function<String, CdsModel> getStrToModel() {
			return strToModel;
		}

		public static class Builder {
			private SidecarAccess sidecarAccess;
			private CacheParams cacheParams;
			private EdmxModelCreator<?> strToEdmx;
			private Function<String, CdsModel> strToModel;

			public Builder sidecarAccess(SidecarAccess sidecarAccess) {
				this.sidecarAccess = sidecarAccess;
				return this;
			}

			public Builder cacheParams(CacheParams cacheParams) {
				this.cacheParams = cacheParams;
				return this;
			}

			public Builder strToEdmx(EdmxModelCreator<?> strToEdmx) {
				this.strToEdmx = strToEdmx;
				return this;
			}

			public Builder strToModel(Function<String, CdsModel> strToModel) {
				this.strToModel = strToModel;
				return this;
			}

			public MetaDataAccessorConfig build() {
				MetaDataAccessorConfig config = new MetaDataAccessorConfig();
				config.sidecarAccess = sidecarAccess;
				config.cacheParams = cacheParams;
				config.strToEdmx = strToEdmx;
				config.strToModel = strToModel;
				return config;
			}
		}
	}

	/**
	 * @param config        MetaDataAccessor configurations
	 * @param cacheTicker   Optional ticker used by guava cache for testing
	 *                      purposes, use null for productive use
	 */
	@SuppressWarnings("unchecked")
	public MetaDataAccessorImpl(MetaDataAccessorConfig config, Ticker cacheTicker) {
		if (cacheTicker == null) {
			cacheTicker = Ticker.systemTicker();
		}
		ExecutorService executorService = Executors.newSingleThreadExecutor();

		SidecarAccess sidecarAccess = config.getSidecarAccess();
		CacheParams cacheParams = config.getCacheParams();
		EdmxModelCreator<M> strToEdmx = (EdmxModelCreator<M>) config.getStrToEdmx();
		Function<String, CdsModel> strToModel = config.getStrToModel();

		if (strToModel == null) {
			modelIdToCdsModel = null;
		} else {
			modelIdToCdsModel = new CdsCache(sidecarAccess, strToModel, cacheParams, cacheTicker, executorService);
		}
		if (strToEdmx == null) {
			modelIdToEdmxModel = null;
		} else {
			modelIdToEdmxModel = new EdmxCache<>(sidecarAccess, strToEdmx, cacheParams, cacheTicker, executorService);
		}
	}

	/**
	 * @param sidecarAccess Object of type {@link SidecarAccess}that provides access
	 *                      to the node.js application sidecar/mtx via a rest API
	 * @param cacheParams   Parameters that control size and lifecycle of cache for
	 *                      cds and edmx models
	 * @param strToEdmx     Function that converts an edmx model description given
	 *                      as string into an edmx model
	 * @param cacheTicker   Optional ticker used by guava cache for testing
	 *                      purposes, use null for productive use
	 */
	public MetaDataAccessorImpl(SidecarAccess sidecarAccess, CacheParams cacheParams, EdmxModelCreator<M> strToEdmx,
			Ticker cacheTicker) {
		this(new MetaDataAccessorConfig.Builder().sidecarAccess(sidecarAccess).cacheParams(cacheParams)
				.strToEdmx(strToEdmx).strToModel(csn -> CdsModelReader.read(
					new CdsModelReader.Config.Builder().setIncludeUIAnnotations(true).build(),
					csn, true)).build(), cacheTicker);
	}

	@Override
	public CdsModel getCdsModel(ModelId key, int maxAgeSeconds) {
		if (modelIdToCdsModel == null) {
			throw new CdsException("Cache not configured");
		}
		return modelIdToCdsModel.getOrLoadIfStale(key, maxAgeSeconds);
	}

	@Override
	public M getEdmx(ModelId key, int maxAgeSeconds) throws CdsException {
		if (modelIdToEdmxModel == null) {
			throw new CdsException("Cache not configured");
		}
		return modelIdToEdmxModel.getOrLoadIfStale(key, maxAgeSeconds);
	}

	@Override
	public void evict(String tenantId) {
		if (modelIdToCdsModel != null) {
			modelIdToCdsModel.evict(tenantId);
		}
		if (modelIdToEdmxModel != null) {
			modelIdToEdmxModel.evict(tenantId);
		}
		// inform listeners
	}

	@Override
	public void refresh(String tenantId, int maxAgeSeconds) {
		if (modelIdToCdsModel != null) {
			modelIdToCdsModel.refresh(tenantId, maxAgeSeconds);
		} if (modelIdToEdmxModel != null) {
			modelIdToEdmxModel.refresh(tenantId, maxAgeSeconds);
		}
		// inform listeners
	}

	private static abstract class Cache<V> {
		private final Ticker ticker;
		private final LoadingCache<ModelId, Entry<V>> cache;
		private final String cacheName = getClass().getSimpleName();

		protected Cache(CacheParams params, Ticker ticker, ExecutorService executorService) {
			this.ticker = ticker;
			this.cache = Caffeine.newBuilder().maximumSize(params.getMaximumSize())
					.expireAfterAccess(params.getExpirationDuration(), params.getExpirationDurationUnit())
					.refreshAfterWrite(params.getRefreshDuration(), params.getRefreshDurationUnit())
					.executor(executorService)
					.ticker(ticker)
					.evictionListener((k, v, c) -> {
						if (c.wasEvicted()) {
							logger.debug("Evicted '{}' in cache '{}' with cause '{}'", k, cacheName, c);
						}
					})
					.build(new CacheLoader<ModelId, Entry<V>>() {
						@Override
						public Entry<V> load(ModelId key) {
							return Cache.this.load(key, null);
						}

						@Override
						public Entry<V> reload(ModelId key, Entry<V> oldValue) {
							logger.debug("Reloading '{}' in cache '{}'", key, cacheName);
							try {
								return Cache.this.load(key, oldValue);
							} catch (Exception e) {// NOSONAR
								logger.error("Reloading '{}' failed", key, e);
								return oldValue;
							}
						}
					});
		}

		public void evict(String tenantId) {
			logger.debug("Evicting tenant '{}' from cache '{}'", tenantId, cacheName);
			forTenant(tenantId, cache::invalidate);
		}

		public void refresh(String tenantId, int maxAgeSeconds) {
			logger.debug("Refreshing tenant '{}' in cache '{}'", tenantId, cacheName);
			forTenant(tenantId, k -> getOrLoadIfStale(k, maxAgeSeconds));
		}

		private void forTenant(String tenantId, Consumer<ModelId> action) {
			cache.asMap().keySet().stream().filter(k -> Objects.equals(tenantId, k.getTenantId())).forEach(action);
		}

		public V getOrLoadIfStale(ModelId key, int maxAgeSeconds) {
			long maxAgeNanos = maxAgeSeconds * NANOS_TO_SECONDS;
			Entry<V> entry;
			try {
				entry = cache.get(key);
			} catch (RuntimeException e) {
				throw new CdsException(e);
			}
			if ((ticker.read() - entry.refreshed()) > maxAgeNanos) {
				// sync load
				Entry<V> loaded = load(key, entry);
				if (loaded != entry) {
					cache.put(key, loaded);
					entry = loaded;
				}
			} else {
				logger.debug("'{}' in cache '{}' is not older than '{}'", key, cacheName, maxAgeSeconds);
			}

			return entry.getEntry();
		}

		private Entry<V> load(ModelId key, Entry<V> oldEntry) {
			logger.debug("Loading '{}' in cache '{}'", key, cacheName);
			String eTag = oldEntry != null ? oldEntry.getETag() : null;
			long beforeAccess = ticker.read();
			ModelAndInformation model = access(key, eTag);
			if (oldEntry != null && model.isNotModified()) {
				oldEntry.refresh(beforeAccess);
				logger.debug("Refreshed unchanged '{}' in cache '{}'", key, cacheName);
				return oldEntry;
			}

			// model has changed -> notify
			return new Entry<>(parse(key, model.getModel()), model.getETag(), beforeAccess);
		}

		abstract ModelAndInformation access(ModelId key, String eTag);

		abstract V parse(ModelId key, String model);

	}

	private static class Entry<V> {
		private final V entry;
		private final String eTag;
		private final AtomicLong refreshed;

		public Entry(V entry, String eTag, long refreshed) {
			this.entry = entry;
			this.eTag = eTag != null ? eTag.trim() : null;
			this.refreshed = new AtomicLong(refreshed);
		}

		public V getEntry() {
			return entry;
		}

		public String getETag() {
			return eTag;
		}

		public void refresh(long refreshed) {
			this.refreshed.set(refreshed);
		}

		public long refreshed() {
			return refreshed.get();
		}

	}

	private static class EdmxCache<M> extends Cache<M> {

		private final SidecarAccess sidecarAccess;
		private final EdmxModelCreator<M> strToEdmx;

		public EdmxCache(SidecarAccess sidecarAccess, EdmxModelCreator<M> strToEdmx, CacheParams params, Ticker ticker, ExecutorService executorService) {
			super(params, ticker, executorService);
			this.sidecarAccess = sidecarAccess;
			this.strToEdmx = strToEdmx;
		}

		@Override
		ModelAndInformation access(ModelId key, String eTag) {
			return sidecarAccess.getEdmx(key, eTag);
		}

		@Override
		M parse(ModelId key, String model) {
			return strToEdmx.parse(model, key.getServiceName().orElse(null));
		}

	}

	private static class CdsCache extends Cache<CdsModel> {

		private final SidecarAccess sidecarAccess;
		private final Function<String, CdsModel> strToModel;

		public CdsCache(SidecarAccess sidecarAccess, Function<String, CdsModel> strToModel, CacheParams params,
				Ticker ticker, ExecutorService executorService) {
			super(params, ticker, executorService);
			this.sidecarAccess = sidecarAccess;
			this.strToModel = strToModel;
		}

		@Override
		ModelAndInformation access(ModelId key, String eTag) {
			return sidecarAccess.getCsn(key, eTag);
		}

		@Override
		CdsModel parse(ModelId key, String csn) {
			return strToModel.apply(csn);
		}

	}

}
