/*
 * ----------------------------------------------------------------
 * © 2019-2021 SAP SE or an SAP affiliate company. All rights reserved.
 * ----------------------------------------------------------------
 *
 */

package com.sap.cds.mtx.impl;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;

import com.sap.cds.mtx.ModelId;
import com.sap.cloud.mt.tools.api.RequestEnhancer;
import com.sap.cloud.mt.tools.api.ResilienceConfig;
import com.sap.cloud.mt.tools.api.ServiceCall;
import com.sap.cloud.mt.tools.api.ServiceResponse;
import com.sap.cloud.mt.tools.exception.InternalException;

/**
 * Class that is responsible for communication with external model provider
 * service
 */

public class ModelProviderAccess extends AbstractSidecarAccess implements SidecarAccess {
    public static final String MODEL_PROVIDER_DESTINATION = "com.sap.cds.modelProvider";
    protected static final String BASE_PATH = "/-/cds/model-provider/";
    private static final String CSN_PATH = BASE_PATH + "getCsn";
    private static final String EDMX_PATH = BASE_PATH + "getEdmx";
    private static final String I18N_PATH = BASE_PATH + "getI18n";

    /**
     * @param requestEnhancer  Adds information to http request, for example for authentication and authority
     * @param resilienceConfig Parameters like number of retries, wait between
     *                         retries, ..
     */
    public ModelProviderAccess(RequestEnhancer requestEnhancer, ResilienceConfig resilienceConfig) {
        super(requestEnhancer, resilienceConfig);
    }

    @Override
    protected String getCsnPath() {
        return CSN_PATH;
    }

    @Override
    protected String getDestinationName() {
        return MODEL_PROVIDER_DESTINATION;
    }

    @Override
    protected Map<String, String> getHeaderFields(String eTag) {
        Map<String, String> headers = new HashMap<>();
        headers.put(HttpHeaders.ACCEPT, APPLICATION_JSON);
        if (Objects.nonNull(eTag)) {
            headers.put(HttpHeaders.IF_NONE_MATCH, eTag);
        }
        return headers;
    }

    @Override
    protected SidecarResponse csnSidecarResponse(ServiceResponse<String> response, String oldHash)
            throws InternalException {
        return modelInfo(response, oldHash);
    }

    @Override
    protected String getEdmxPath() {
        return EDMX_PATH;
    }

    @Override
    protected SidecarResponse edmxSidecarResponse(ServiceResponse<String> response, String oldHash)
            throws InternalException {
        return modelInfo(response, oldHash);
    }

    @Override
    protected String getI18nPath() {
        return I18N_PATH;
    }

    @Override
    protected SidecarResponse i18nSidecarResponse(ServiceResponse<String> response, String oldHash)
            throws InternalException {
        return modelInfo(response, oldHash);
    }

    @Override
    protected ServiceCall createCsnCall(ModelId id, String eTag) throws InternalException {
        return csnEndpoint.createServiceCall()
                .http()
                .post()
                .payload(getCsnBody(id))
                .noPathParameter()
                .noQuery()
                .enhancer(requestEnhancer)
                .insertHeaderFields(getHeaderFields(eTag))
                .end();
    }

    @Override
    protected ServiceCall createEdmxCall(ModelId id, String eTag) throws InternalException {
        return edmxEndpoint.createServiceCall()
                .http()
                .post()
                .payload(getEdmxBody(id))
                .noPathParameter()
                .noQuery()
                .enhancer(requestEnhancer)
                .insertHeaderFields(getHeaderFields(eTag))
                .end();
    }

    @Override
    protected ServiceCall createI18nCall(ModelId id, String eTag) throws InternalException {
        return i18nEndpoint.createServiceCall()
                .http()
                .post()
                .payload(getI18nBody(id))
                .noPathParameter()
                .noQuery()
                .enhancer(requestEnhancer)
                .insertHeaderFields(getHeaderFields(eTag))
                .end();
    }

    private SidecarResponse modelInfo(ServiceResponse<String> response, String oldEtag) throws InternalException {
        if (response.getHttpStatusCode() == HttpStatus.SC_NOT_MODIFIED) {
            return new SidecarResponse(null, oldEtag, true);
        } else {
            String eTag = response.getETag().orElse(null);
            return new SidecarResponse(getPayload(response), eTag, false);
        }
    }


    private String getPayload(ServiceResponse<String> response) throws InternalException {
        return response.getPayload().orElse("");
    }

    protected Map<String, Object> getBody(ModelId id) {
        Map<String, Object> body = new HashMap<>();
        body.put("tenant", id.getTenantId());
        body.put("toggles", id.getFeatures());
        body.put("for", "java");
        return body;
    }

    protected Map<String, Object> getEdmxBody(ModelId id) {
        Map<String, Object> body = getBody(id);
        id.getServiceName().ifPresent(s -> body.put("service", s));
        id.getLanguage().ifPresent(l -> body.put("locale", l));
        id.getODataVersion().ifPresent(v -> body.put("flavor", v));
        return body;
    }

    protected Map<String, Object> getI18nBody(ModelId id) {
        Map<String, Object> body = getBody(id);
        body.put("locale", id.getLanguage().orElse(""));
        return body;
    }

    protected Map<String, Object> getCsnBody(ModelId id) {
        return getBody(id);
    }
}
