/*
 * ----------------------------------------------------------------
 * © 2019-2021 SAP SE or an SAP affiliate company. All rights reserved.
 * ----------------------------------------------------------------
 *
 */

package com.sap.cds.mtx.impl;

import java.util.ArrayList;
import java.util.List;

import com.sap.cds.mtx.ModelId;
import com.sap.cloud.mt.tools.api.QueryParameter;
import com.sap.cloud.mt.tools.api.RequestEnhancer;
import com.sap.cloud.mt.tools.api.ResilienceConfig;
import com.sap.cloud.mt.tools.api.ServiceCall;
import com.sap.cloud.mt.tools.exception.InternalException;

/**
 * Class that is responsible for communication with node.js application
 * sidecar/mtx via its REST API
 */

public class SidecarAccessV1 extends AbstractSidecarAccess implements SidecarAccess {
    public static final String MTX_SIDECAR_DESTINATION = "com.sap.cds.mtxSidecar";
    private static final String CSN_PATH = "/mtx/v1/metadata/csn/";
    private static final String EDMX_PATH = "/mtx/v1/metadata/edmx/";

    /**
     * @param requestEnhancer  Adds information to http request, for example for authentication and authority
     * @param resilienceConfig Parameters like number of retries, wait between
     *                         retries, ..
     */
    public SidecarAccessV1(RequestEnhancer requestEnhancer, ResilienceConfig resilienceConfig) {
        super(requestEnhancer, resilienceConfig);
    }

    @Override
    protected String getDestinationName() {
        return MTX_SIDECAR_DESTINATION;
    }

    @Override
    protected ServiceCall createCsnCall(ModelId id, String eTag) throws InternalException {
        return csnEndpoint.createServiceCall()
                .http()
                .get()
                .withoutPayload()
                .pathParameter(id.getTenantId())
                .noQuery()
                .enhancer(requestEnhancer)
                .insertHeaderFields(getHeaderFields(eTag))
                .end();
    }

    @Override
    protected ServiceCall createEdmxCall(ModelId id, String eTag) throws InternalException {
        return edmxEndpoint.createServiceCall()
                .http()
                .get()
                .withoutPayload()
                .pathParameter(id.getTenantId())
                .query(getQuery(id))
                .enhancer(requestEnhancer)
                .insertHeaderFields(getHeaderFields(eTag))
                .end();
    }

    private List<QueryParameter> getQuery(ModelId id) {
        List<QueryParameter> queryParameters = new ArrayList<>();
        id.getServiceName().ifPresent(s -> queryParameters.add(new QueryParameter("name", s)));
        id.getLanguage().ifPresent(l -> queryParameters.add(new QueryParameter("language", l)));
        id.getODataVersion().ifPresent(v -> queryParameters.add(new QueryParameter("odataVersion", v)));
        return queryParameters;
    }

    @Override
    protected String getCsnPath() {
        return CSN_PATH;
    }

    @Override
    protected String getEdmxPath() {
        return EDMX_PATH;
    }

    @Override
    protected String getI18nPath() {
        // unsupported operation in the old sidecar version
        return null;
    }

    @Override
    protected ServiceCall createI18nCall(ModelId id, String eTag) throws InternalException {
        // unsupported operation in the old sidecar version
        return null;
    }

}
