/*******************************************************************
 * © 2019 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.mtx;

import static com.sap.cds.mtx.ModelId.create;

import java.util.Map;

import com.sap.cds.reflect.CdsModel;

/**
 * Provides access to tenant-specific CDS and EDMX models
 *
 * @param <M> EDMX model type
 */

public interface MetaDataAccessor<M> {
	/**
	 * @param modelId       model identifier
	 * @param maxAgeSeconds maximum acceptable age of resource
	 * @return EDMX model
	 */
	public M getEdmx(ModelId modelId, int maxAgeSeconds);

	/**
	 * @param modelId       model identifier
	 * @param maxAgeSeconds maximum acceptable age of resource
	 * @return model related i18n resources
	 */
	public I18n getI18n(ModelId modelId, int maxAgeSeconds);
	public record I18n(String eTag, Map<String, String> texts) {};

	/**
	 * @param tenantId tenant identifier
	 * @return CDS model
	 */
	default CdsModel getCdsModel(String tenantId) {
		return getCdsModel(create(tenantId).build(), Integer.MAX_VALUE);
	}

	/**
	 * @param modelId       model identifier
	 * @param maxAgeSeconds maximum acceptable age of resource
	 * @return CDS model
	 */
	public CdsModel getCdsModel(ModelId modelId, int maxAgeSeconds);

	/**
	 * Synchronously CDS and EDMX models for the specified tenant if they have
	 * changed
	 *
	 * @param tenantId tenant identifier
	 */
	default void refresh(String tenantId) {
		refresh(tenantId, 0);
	}

	/**
	 * TODO sync Rereads CDS and EDMX models for the specified tenant if they have
	 * changed
	 *
	 * @param tenantId      tenant identifier
	 * @param maxAgeSeconds maximum acceptable age of resource
	 */
	public void refresh(String tenantId, int maxAgeSeconds);

	/**
	 * evicts CDS and EDMX models for the specified tenant
	 *
	 * @param tenantId tenant identifier
	 */
	public void evict(String tenantId);
}
