/*******************************************************************
 * © 2021 SAP SE or an SAP affiliate company. All rights reserved. *
 *******************************************************************/
package com.sap.cds.mtx;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.MoreObjects;
import com.sap.cds.CdsException;

public class ModelId {

	private final String tenantId;
	private final String serviceName;
	private final String language;
	private final String odataVersion;
	private final String version;
	private final Set<String> features;

	private ModelId(String tenantId, String serviceName, String language, String version, String odataVersion,
			Set<String> features) throws CdsException {
		this.tenantId = tenantId;
		this.serviceName = serviceName;
		this.language = language;
		this.odataVersion = odataVersion;
		this.version = version;
		this.features = new HashSet<>(features);
	}

	public String getTenantId() {
		return tenantId;
	}

	public Optional<String> getServiceName() {
		return Optional.ofNullable(serviceName);
	}

	public Optional<String> getLanguage() {
		return Optional.ofNullable(language);
	}

	public String getVersion() {
		return version;
	}

	public Optional<String> getODataVersion() {
		return Optional.ofNullable(odataVersion);
	}

	public Set<String> getFeatures() {
		return features;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o)
			return true;
		if (!(o instanceof ModelId))
			return false;
		ModelId key = (ModelId) o;

		return Objects.equals(tenantId, key.tenantId) && version.equals(key.version) && features.equals(key.features)
				&& Objects.equals(serviceName, key.serviceName) && Objects.equals(language, key.language)
				&& Objects.equals(odataVersion, key.odataVersion);
	}

	@Override
	public int hashCode() {
		return Objects.hash(tenantId, serviceName, language, odataVersion, version, features);
	}

	@Override
	public String toString() {
		return MoreObjects.toStringHelper(this)
				.omitNullValues()
				.add("tenantId", tenantId)
				.add("serviceName", serviceName)
				.add("language", language)
				.add("odataVersion", odataVersion)
				.add("version", version)
				.add("features", features)
				.toString();
	}

	public static Builder create(String tenant) {
		return new Builder(tenant);
	}

	public static class Builder {

		private final String tenant;
		private String service = null;
		private String language = null;
		private String version = "";
		private String odataVersion = null;
		private Set<String> features = Collections.emptySet();

		public Builder(String tenant) {
			this.tenant = tenant;
		}

		public Builder service(String service) {
			this.service = service;

			return this;
		}

		public Builder language(String language) {
			this.language = language;

			return this;
		}

		public Builder odataVersion(String odataVersion) {
			this.odataVersion = odataVersion;

			return this;
		}

		public Builder odata(String service, String language, String odataVersion) {
			this.service = service;
			this.language = language;
			this.odataVersion = odataVersion;

			return this;
		}

		public Builder version(String version) {
			notNull("Version", version);
			this.version = version;

			return this;
		}

		public Builder features(Set<String> features) {
			notNull("Features", features);
			this.features = features;

			return this;
		}

		public Builder features(String... features) {
			notNull("Features", features);
			return features(Arrays.stream(features).collect(Collectors.toSet()));
		}

		public Builder allFeatures() {
			return features("*");
		}

		public ModelId build() {
			return new ModelId(tenant, service, language, version, odataVersion, features);
		}

		private void notNull(String string, Object value) {
			if (value == null) {
				throw new CdsException(string + " must not be null");
			}
		}

	}

}
