/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/
package com.sap.cloud.mt.runtime;

import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Stream;

public class DataPoolSettings {

    private final Map<ConnectionPoolType, ParameterList> parameterLists = new EnumMap<>(ConnectionPoolType.class);
    private final Map<ConnectionPoolType, Map<String, String>> dbProperties = new EnumMap<>(ConnectionPoolType.class);

    public DataPoolSettings(String hikariPrefix, String tomcatPrefix, String atomikosPrefix) {
        this(createLegacyConfig(hikariPrefix, tomcatPrefix, atomikosPrefix));
    }

    private static Map<String, ConnectionPoolType> createLegacyConfig(String hikariPrefix, String tomcatPrefix, String atomikosPrefix) {
        Map<String, ConnectionPoolType> config = new HashMap<>();
        config.put(hikariPrefix, ConnectionPoolType.HIKARI);
        config.put(tomcatPrefix, ConnectionPoolType.TOMCAT);
        config.put(atomikosPrefix, ConnectionPoolType.ATOMIKOS);
        return config;
    }

    /**
     * Constructor
     *
     * @param prefixToPoolType maps a parameter prefix to a {@link ConnectionPoolType}
     */
    public DataPoolSettings(Map<String, ConnectionPoolType> prefixToPoolType) {
        for (ConnectionPoolType type : ConnectionPoolType.values()) {
            parameterLists.put(type, type.createParameterList());
            dbProperties.put(type, new HashMap<>());
        }
        for (Map.Entry<String, ConnectionPoolType> entry : prefixToPoolType.entrySet()) {
            String prefix = entry.getKey();
            parameterLists.get(entry.getValue()).initList(prefix);
        }
    }

    public Stream<Parameter> getHikariParameters() {
        return parameterLists.get(ConnectionPoolType.HIKARI).stream();
    }

    public Stream<Parameter> getTomcatParameters() {
        return parameterLists.get(ConnectionPoolType.TOMCAT).stream();
    }

    public Stream<Parameter> getAtomikosParameters() {
        return parameterLists.get(ConnectionPoolType.ATOMIKOS).stream();
    }

    public static class Parameter {
        private final String name;
        private final Class<?> type;
        private final String prefix;

        private Parameter(String name, Class<?> type, String prefix) {
            this.name = name;
            this.type = type;
            this.prefix = prefix;
        }

        public String getNameInEnv() {
            return prefix + "." + name;
        }

        public String getPoolPropertyName() {
            return name;
        }

        public Class<?> getType() {
            return type;
        }

        public String getSetterName() {
            return "set" + name.substring(0, 1).toUpperCase(Locale.ENGLISH) + name.substring(1);
        }
    }

    public enum ConnectionPoolType {
        HIKARI(() -> new HikariParameterList()),
        TOMCAT(() -> new TomcatParameterList()),
        ATOMIKOS(() -> new AtomikosParameterList());

        private final Supplier<ParameterList> paramListSupplier;

        private ConnectionPoolType(Supplier<ParameterList> paramListSupplier) {
            this.paramListSupplier = paramListSupplier;
        }

        ParameterList createParameterList() {
            return paramListSupplier.get();
        }
    }

    public interface ParameterList extends List<Parameter> {

        void initList(String prefix);

    }

    @SuppressWarnings("serial")
    private static class HikariParameterList extends LinkedList<Parameter> implements ParameterList {

        @Override
        public void initList(String prefix) {
            add(new Parameter("autoCommit", boolean.class, prefix));
            add(new Parameter("connectionTimeout", long.class, prefix));
            add(new Parameter("idleTimeout", long.class, prefix));
            add(new Parameter("maxLifetime", long.class, prefix));
            add(new Parameter("minimumIdle", int.class, prefix));
            add(new Parameter("maximumPoolSize", int.class, prefix));
            add(new Parameter("initializationFailTimeout", long.class, prefix));
            add(new Parameter("isolateInternalQueries", boolean.class, prefix));
            add(new Parameter("allowPoolSuspension", boolean.class, prefix));
            add(new Parameter("readOnly", boolean.class, prefix));
            add(new Parameter("registerMbeans", boolean.class, prefix));
            add(new Parameter("catalog", String.class, prefix));
            add(new Parameter("connectionInitSql", String.class, prefix));
            add(new Parameter("transactionIsolation", String.class, prefix));
            add(new Parameter("validationTimeout", long.class, prefix));
            add(new Parameter("leakDetectionThreshold", long.class, prefix));
            add(new Parameter("connectionTestQuery", String.class, prefix));
        }
    }

    @SuppressWarnings("serial")
    private static class TomcatParameterList extends LinkedList<Parameter> implements ParameterList {

        @Override
        public void initList(String prefix) {
            add(new Parameter("initialSize", int.class, prefix));
            add(new Parameter("maxWait", int.class, prefix));
            add(new Parameter("maxActive", int.class, prefix));
            add(new Parameter("maxIdle", int.class, prefix));
            add(new Parameter("minIdle", int.class, prefix));
            add(new Parameter("defaultAutoCommit", boolean.class, prefix));
            add(new Parameter("testOnBorrow", boolean.class, prefix));
            add(new Parameter("validationInterval", long.class, prefix));
            add(new Parameter("validationQueryTimeout", int.class, prefix));
            add(new Parameter("validationQuery", String.class, prefix));
            add(new Parameter("defaultReadOnly", boolean.class, prefix));
            add(new Parameter("testOnConnect", boolean.class, prefix));
            add(new Parameter("testOnReturn", boolean.class, prefix));
            add(new Parameter("testWhileIdle", boolean.class, prefix));
            add(new Parameter("timeBetweenEvictionRunsMillis", int.class, prefix));
            add(new Parameter("minEvictableIdleTimeMillis", int.class, prefix));
            add(new Parameter("removeAbandoned", boolean.class, prefix));
            add(new Parameter("removeAbandonedTimeout", int.class, prefix));
            add(new Parameter("logAbandoned", boolean.class, prefix));
            add(new Parameter("initSQL", String.class, prefix));
            add(new Parameter("jdbcInterceptors", String.class, prefix));
            add(new Parameter("jmxEnabled", boolean.class, prefix));
            add(new Parameter("fairQueue", boolean.class, prefix));
            add(new Parameter("abandonWhenPercentageFull", int.class, prefix));
            add(new Parameter("maxAge", long.class, prefix));
            add(new Parameter("useEquals", boolean.class, prefix));
            add(new Parameter("suspectTimeout", int.class, prefix));
            add(new Parameter("rollbackOnReturn", boolean.class, prefix));
            add(new Parameter("commitOnReturn", boolean.class, prefix));
            add(new Parameter("useDisposableConnectionFacade", boolean.class, prefix));
            add(new Parameter("logValidationErrors", boolean.class, prefix));
            add(new Parameter("propagateInterruptState", boolean.class, prefix));
            add(new Parameter("ignoreExceptionOnPreLoad", boolean.class, prefix));
            add(new Parameter("useStatementFacade", boolean.class, prefix));
        }
    }

    @SuppressWarnings("serial")
    private static class AtomikosParameterList extends LinkedList<Parameter> implements ParameterList {

        @Override
        public void initList(String prefix) {
            add(new Parameter("maintenanceInterval", int.class, prefix));
            add(new Parameter("maxIdleTime", int.class, prefix));
            add(new Parameter("maxPoolSize", int.class, prefix));
            add(new Parameter("minPoolSize", int.class, prefix));
            add(new Parameter("reapTimeout", int.class, prefix));
            add(new Parameter("testQuery", String.class, prefix));
            add(new Parameter("borrowConnectionTimeout", int.class, prefix));
        }
    }

    public Properties getDbProperties(ConnectionPoolType type) {
        Properties properties = new Properties();
        if (dbProperties.get(type) != null) {
            dbProperties.get(type).forEach(properties::put);
        }
        return properties;
    }

    public void setDbProperties(ConnectionPoolType type, Map<String, String> dbProperties) {
        if (dbProperties != null) {
            this.dbProperties.put(type, dbProperties);
        }
    }
}
