/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/

package com.sap.cloud.mt.runtime;

import com.sap.cloud.mt.subscription.*;
import com.sap.cloud.mt.subscription.exceptions.InternalError;
import com.sap.cloud.mt.subscription.exceptions.UnknownTenant;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

public abstract class DataSourceLookup {
    private static final Logger logger = LoggerFactory.getLogger(DataSourceLookup.class);
    private final ConcurrentHashMap<String, DataSourceAndInfo> tenantToDataSource = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, DataSource> dbKeyToDataSource = new ConcurrentHashMap<>();
    private final InstanceLifecycleManager instanceLifecycleManager;
    private boolean oneDataSourcePerDb = false;

    public DataSourceLookup(InstanceLifecycleManager instanceLifecycleManager, boolean oneDataSourcePerDb) {
        this.instanceLifecycleManager = instanceLifecycleManager;
        this.oneDataSourcePerDb = oneDataSourcePerDb;
    }

    public DataSourceLookup(InstanceLifecycleManager instanceLifecycleManager) {
        this(instanceLifecycleManager, false);
    }

    DataSourceAndInfo getDataSourceAndInfo(String tenantId) throws InternalError, UnknownTenant {
        DataSourceAndInfo dataSourceAndInfo = tenantToDataSource.get(tenantId);
        if (dataSourceAndInfo != null) {
            return dataSourceAndInfo;
        } else {
            synchronized (TenantMutexFactory.get(tenantId)) {
                dataSourceAndInfo = tenantToDataSource.get(tenantId);
                if (dataSourceAndInfo != null) {
                    return dataSourceAndInfo;
                }
                logger.debug("Access instance manager for tenant {}", tenantId);
                DataSourceInfo info = instanceLifecycleManager.getDataSourceInfo(tenantId, true);
                String dbKey = DbUtils.getDbKey(info);
                DataSource dataSource = null;
                if (oneDataSourcePerDb) {
                    dataSource = dbKeyToDataSource.get(dbKey);
                    if (dataSource == null) {
                        preparePools();
                        dataSource = dbKeyToDataSource.get(dbKey);
                        if (dataSource == null) {
                            throw new InternalError("Could not find database pool for db key " + dbKey);
                        }
                    }
                } else {
                    //make sure that also in mode "one pool per tenant" lib dummy containers are created
                    preparePools();
                    dataSource = create(info);
                }
                DataSourceAndInfo newEntry = new DataSourceAndInfo(dataSource, info);
                tenantToDataSource.put(tenantId, newEntry);
                return newEntry;
            }
        }
    }

    private void preparePools() throws InternalError {
        preparePools(false);
    }

    private void preparePools(boolean force) throws InternalError {
        if (!force && dbKeyToDataSource.size() > 0) return;
        List<DataSourceInfo> libContainers = instanceLifecycleManager.createAndGetLibContainers();
        libContainers.stream().forEach(info -> {
            String dbKey = DbUtils.getDbKey(info);
            if (!dbKeyToDataSource.containsKey(dbKey)) {
                try {
                    DataSource dataSource = create(info);
                    dbKeyToDataSource.put(dbKey, dataSource);
                } catch(InternalError internalError) {
                    logger.error("Cannot create data source pool for tenant {}", info.getTenantId());
                }
            }
        });
    }

    boolean fixDataSourceAfterCredentialChange(String tenantId) throws InternalError {
        if (tenantId == null || tenantId.isEmpty()) {
            return false;
        }
        synchronized (TenantMutexFactory.get(tenantId)) {
            if (!oneDataSourcePerDb) {
                // Refresh only, if the container was really recreated for the same tenant
                // A parallel task could have fixed it already.
                // It must be avoided that pools that were fixed and are used are accidentally deleted
                DataSourceAndInfo dataSourceAndInfo = tenantToDataSource.get(tenantId);
                if (dataSourceAndInfo == null) {
                    //already removed from cache
                    return true;
                }
                DataSourceInfo cachedInfo = dataSourceAndInfo.getDataSourceInfo();
                DataSourceInfo currentInfo;
                try {
                    currentInfo = instanceLifecycleManager.getDataSourceInfo(tenantId, true);
                } catch(UnknownTenant unknownTenant) {
                    currentInfo = null;
                }
                if (currentInfo == null ||
                        !StringUtils.equals(currentInfo.getPassword(), cachedInfo.getPassword()) ||
                        !StringUtils.equals(currentInfo.getUser(), cachedInfo.getUser()) ||
                        !StringUtils.equals(currentInfo.getSchema(), cachedInfo.getSchema()) ||
                        !StringUtils.equals(currentInfo.getDbKey(), cachedInfo.getDbKey())) {
                    deleteFromCache(tenantId);
                    return true;
                } else {
                    // credentials haven't changed => nothing to fix
                    return false;
                }
            } else {
                // in mode oneDataSourceDB, invalid connections are fixed by the pool. It retrieves connections for mt-lib's own
                // schemas. These schemas are never deleted.
                return false;
            }
        }
    }

    void deleteFromCache(String tenantId) {
        if (tenantId == null) {
            return;
        }
        logger.debug("Instance for tenant {} deleted from cache", tenantId);
        synchronized (TenantMutexFactory.get(tenantId)) {
            DataSourceAndInfo dataSourceAndInfo = tenantToDataSource.get(tenantId);
            if (dataSourceAndInfo == null) {
                return;
            }
            tenantToDataSource.remove(tenantId);
            if (oneDataSourcePerDb) {
                long numberOfEntries = tenantToDataSource.entrySet().stream().filter(e ->
                        e.getValue().getDataSource() == dataSourceAndInfo.getDataSource()).count();
                if (numberOfEntries == 0) {
                    dbKeyToDataSource.remove(DbUtils.getDbKey(dataSourceAndInfo.getDataSourceInfo()));
                    closeDataSource(dataSourceAndInfo.getDataSource());
                }
            } else {
                closeDataSource(dataSourceAndInfo.getDataSource());
            }
        }
    }

    public void reset() {
        if (oneDataSourcePerDb) {
            tenantToDataSource.clear();
            dbKeyToDataSource.entrySet().stream().forEach(e -> closeDataSource(e.getValue()));
            dbKeyToDataSource.clear();
        } else {
            tenantToDataSource.entrySet().stream().forEach(e -> closeDataSource(e.getValue().getDataSource()));
            tenantToDataSource.clear();
        }
    }

    public List<DataSourceInfo> getCachedDataSource() {
        List<DataSourceInfo> result = new ArrayList<>();
        tenantToDataSource.entrySet().stream().forEach(e -> result.add(e.getValue().getDataSourceInfo().clone()));
        return result;
    }

    public List<HealtCheckResult> checkDataSourcePerDb(String dummySelectStatement) {
        List<HealtCheckResult> result = new ArrayList<>();
        try {
            preparePools();
        } catch(InternalError internalError) {
            HealtCheckResult healtCheckResult = new HealtCheckResult("", false, internalError);
            result.add(healtCheckResult);
            return result;
        }
        dbKeyToDataSource.entrySet().stream()
                .forEach(e -> {
                    try (Connection connection = e.getValue().getConnection()) {
                        SqlOperations sqlOperations = SqlOperations.build(instanceLifecycleManager.getDbIdentifiers().getDB());
                        sqlOperations.setDummySelectStatement(dummySelectStatement);
                        sqlOperations.dummySelect(connection);
                        HealtCheckResult healtCheckResult = new HealtCheckResult(e.getKey(), true, null);
                        result.add(healtCheckResult);
                    } catch(InternalError | SQLException exception) {
                        HealtCheckResult healtCheckResult = new HealtCheckResult(e.getKey(), false, exception);
                        result.add(healtCheckResult);
                    }
                });
        return result;
    }

    public void checkDataSource(String tenantId, String dummySelectStatement) throws SQLException {
        DataSourceAndInfo dataSourceAndInfo = tenantToDataSource.get(tenantId);
        if (dataSourceAndInfo == null) {
            throw new SQLException("Tenant " + tenantId + " doesn't exist");
        } else {
            try (Connection connection = dataSourceAndInfo.getDataSource().getConnection()) {
                SqlOperations sqlOperations = SqlOperations.build(instanceLifecycleManager.getDbIdentifiers().getDB());
                sqlOperations.setDummySelectStatement(dummySelectStatement);
                sqlOperations.dummySelect(connection);
            } catch(InternalError exception) {
                throw new SQLException(exception);
            }
        }
    }

    boolean doesTenantExist(String tenantId) {
        try {
            instanceLifecycleManager.checkThatTenantExists(tenantId);
            return true;
        } catch(UnknownTenant unknownTenant) {
            return false;
        }
    }

    protected abstract DataSource create(DataSourceInfo info) throws InternalError;

    protected abstract void closeDataSource(DataSource dataSource);

    public boolean isOneDataSourcePerDb() {
        return oneDataSourcePerDb;
    }

    public DbIdentifiers.DB getDbType() {
        return instanceLifecycleManager.getDbType();
    }

    public boolean hasDbIdentifiers() {
        return instanceLifecycleManager.hasDbIdentifiers();
    }
}