/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/
package com.sap.cloud.mt.runtime;

import com.sap.cloud.mt.subscription.DataSourceInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.sql.SQLException;
import java.util.*;


public class DbHealthIndicatorImpl<T> {
    private static final Logger logger = LoggerFactory.getLogger(DbHealthIndicatorImpl.class);
    private static final int STACK_TRACE_WRITE_PERIOD = 100;
    private final String healthDummySelect;
    private final TenantAwareDataSource dataSource;
    private final ConnectionChecker connectionChecker;
    private long lastChecked = 0;
    private T lastHealth = null;
    private final Long healthCheckIntervalMillis;
    private final HealthUp<T> healthUp;
    private final HealthDown<T> healthDown;
    private final HealthDownDetails<T> healthDownDetails;
    private final HealthUpDetails<T> healthUpDetails;
    private final boolean newCheck;
    private int callCounter = 0;

    public DbHealthIndicatorImpl(String healthDummySelect, TenantAwareDataSource dataSource,
                                 Long healthCheckIntervalMillis,
                                 HealthUp<T> healthUp, HealthDown<T> healthDown, HealthDownDetails<T> healthDownDetails,
                                 HealthUpDetails<T> healthUpDetails,
                                 boolean newCheck) {
        this.healthDummySelect = healthDummySelect;
        this.dataSource = dataSource;
        this.connectionChecker = new ConnectionChecker();
        this.healthCheckIntervalMillis = healthCheckIntervalMillis;
        this.healthUp = healthUp;
        this.healthDown = healthDown;
        this.healthDownDetails = healthDownDetails;
        this.healthUpDetails = healthUpDetails;
        this.newCheck = newCheck;
    }

    public DbHealthIndicatorImpl(String healthDummySelect, TenantAwareDataSource dataSource,
                                 Long healthCheckIntervalMillis,
                                 HealthUp<T> healthUp, HealthDown<T> healthDown, HealthDownDetails<T> healthDownDetails) {
        this(healthDummySelect, dataSource, healthCheckIntervalMillis, healthUp, healthDown, healthDownDetails,
                null, false);
    }

    public synchronized T health() {
        callCounter++;
        if (callCounter > STACK_TRACE_WRITE_PERIOD) {
            callCounter = 0;
        }
        long currentTime = System.currentTimeMillis();
        if (lastChecked != 0 && ((currentTime - lastChecked) < healthCheckIntervalMillis)) {
            return lastHealth;
        }
        lastChecked = currentTime;
        try {
            List<String> detailInfo = new ArrayList<>();
            boolean down = false;
            if (newCheck) {
                down = isDownNewCheck(detailInfo);
            } else {
                List<DataSourceInfo> oneInfoPerDb = filterOneInfoPerDB(dataSource.getDataSourceLookup().getCachedDataSource());
                down = isDown(detailInfo, oneInfoPerDb);
            }
            if (down) {
                lastHealth = healthDownDetails.execute("Detail information:", detailInfo);
                return lastHealth;
            } else {
                lastHealth = healthUpDetails != null ?
                        healthUpDetails.execute("Detail information:", detailInfo) : healthUp.execute();
                return lastHealth;
            }
        } catch (RuntimeException e) {
            lastHealth = healthDown.execute();
            return lastHealth;
        }
    }

    private boolean isDown(List<String> detailInfo, List<DataSourceInfo> oneInfoPerDb) {
        Boolean[] downArray = {false};
        oneInfoPerDb.stream().forEach(info -> {
            if (info.getTenantId() != null && !info.getTenantId().trim().isEmpty()) {
                try {
                    connectionChecker.checkConnection(info.getTenantId(), dataSource, healthDummySelect);
                    detailInfo.add("Connection for DB " + info.getHost() + ":" + info.getPort() + " is ok");
                } catch (SQLException e) {
                    logger.error("Could not open connection for DB {}", info.getHost() + ":" + info.getPort());
                    logger.debug("The following error was reported: {}", e.getMessage());
                    detailInfo.add("Could not open connection for DB " + info.getHost() + ":" + info.getPort());
                    downArray[0] = true;
                }
            }
        });
        return downArray[0];
    }

    private boolean isDownNewCheck(List<String> detailInfo) {
        List<HealtCheckResult> healthCheckResults = dataSource.getDataSourceLookup().checkDataSourcePerDb(healthDummySelect);
        Boolean[] down = {false};
        healthCheckResults.stream().forEach(result -> {
            if (!result.isOk()) {
                logger.error("Could not open connection for DB {}", result.getDbIdentifier());
                if (callCounter == STACK_TRACE_WRITE_PERIOD) {
                    logger.error("The following error was reported: ", result.getException());
                } else {
                    logger.error("The following error was reported: {}", result.getException().getMessage());
                }
                detailInfo.add("Could not open connection for DB " + result.getDbIdentifier() + ". Error is: " +
                        result.getException().getMessage());
                down[0] = true;
            } else {
                logger.debug("Connection for DB {} is ok", result.getDbIdentifier());
                detailInfo.add("Connection for DB " + result.getDbIdentifier() + " is ok");
            }
        });
        return down[0];
    }

    public String getHealthDummySelect() {
        return healthDummySelect;
    }

    private List<DataSourceInfo> filterOneInfoPerDB(List<DataSourceInfo> infoList) {
        Map<String, List<DataSourceInfo>> urlToInfo = new HashMap<>();
        infoList.stream().forEach(i -> {
            List<DataSourceInfo> list = urlToInfo.get(DbUtils.getDbKey(i));
            if (list == null) {
                list = new ArrayList<>();
                urlToInfo.put(DbUtils.getDbKey(i), list);
            }
            list.add(i);
        });
        urlToInfo.entrySet().stream().forEach(e -> e.getValue().sort(new Comparator<DataSourceInfo>() {
            @Override
            public int compare(DataSourceInfo o1, DataSourceInfo o2) {
                return o1.getTenantId().compareTo(o2.getTenantId());
            }
        }));
        List<DataSourceInfo> filteredList = new ArrayList<>();
        urlToInfo.entrySet().stream().forEach(e -> filteredList.add(e.getValue().get(0)));
        return filteredList;
    }

    @FunctionalInterface
    public interface HealthUp<T> {
        public T execute();
    }

    @FunctionalInterface
    public interface HealthDown<T> {
        public T execute();
    }

    @FunctionalInterface
    public interface HealthDownDetails<T> {
        public T execute(String text, List<String> detailInfo);
    }

    @FunctionalInterface
    public interface HealthUpDetails<T> {
        public T execute(String text, List<String> detailInfo);
    }
}