/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/

package com.sap.cloud.mt.runtime;

import com.sap.cloud.mt.subscription.InstanceLifecycleManager;

import java.util.Locale;
import java.util.Map;

public class DataSourceLookupBuilder {
    private static final String TOMCAT = "tomcat";
    private static final String HIKARI = "hikari";
    private InstanceLifecycleManager instanceLifecycleManager;
    private EnvironmentAccess environmentAccess;
    private String poolProvider;
    private boolean oneDataSourcePerDb;
    private boolean registerJmx;
    private Map<String, DataPoolSettings.ConnectionPoolType> prefixToPoolType;

    private DataSourceLookupBuilder() {
    }

    public static DataSourceLookupBuilder create() {
        return new DataSourceLookupBuilder();
    }

    public DataSourceLookup build() {
        DataPoolSettings dataPoolSettings;
        if (prefixToPoolType != null && !prefixToPoolType.isEmpty()) {
            dataPoolSettings = new DataPoolSettings(prefixToPoolType);
        } else {
            dataPoolSettings = new DataPoolSettings("spring.datasource.hikari", "spring.datasource.tomcat",
                    "com.sap.mt.atomikos");
        }
        if (poolProvider == null || poolProvider.isEmpty()) {
            poolProvider = HIKARI;
        }
        switch (poolProvider.toLowerCase(Locale.ENGLISH)) {
            case TOMCAT:
                return new TomcatDataSourceLookup(instanceLifecycleManager, environmentAccess, dataPoolSettings,
                        oneDataSourcePerDb, registerJmx);
            case HIKARI:
                return new HikariDataSourceLookup(instanceLifecycleManager, environmentAccess, dataPoolSettings,
                        oneDataSourcePerDb);
            default:
                throw new IllegalArgumentException("Pool provider " + poolProvider + " is not supported");
        }
    }

    public DataSourceLookupBuilder instanceLifecycleManager(InstanceLifecycleManager instanceLifecycleManager) {
        this.instanceLifecycleManager = instanceLifecycleManager;
        return this;
    }

    public DataSourceLookupBuilder environmentAccess(EnvironmentAccess environmentAccess) {
        this.environmentAccess = environmentAccess;
        return this;
    }

    public DataSourceLookupBuilder poolProvider(String poolProvider) {
        this.poolProvider = poolProvider;
        return this;
    }

    public DataSourceLookupBuilder oneDataSourcePerDb(boolean oneDataSourcePerDb) {
        this.oneDataSourcePerDb = oneDataSourcePerDb;
        return this;
    }

    public DataSourceLookupBuilder registerJmx(boolean registerJmx) {
        this.registerJmx = registerJmx;
        return this;
    }

    public DataSourceLookupBuilder prefixToPoolType(Map<String, DataPoolSettings.ConnectionPoolType> prefixToPoolType) {
        this.prefixToPoolType = prefixToPoolType;
        return this;
    }
}
