/*
 * *************************************************************************
 *  * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 *  *************************************************************************
 */

package com.sap.cloud.mt.subscription;

import com.sap.cloud.mt.subscription.exceptions.InternalError;
import org.apache.commons.lang3.StringUtils;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

public class DbCredentialsBuilder {
    private static final String DBNAME = "dbname";
    private static final String DATABASE = "database";
    private static final String URI = "uri";
    private static final String JDBC = "jdbc:";
    private static final String POSTGRES = "postgres:";
    private static final String POSTGRESQL = "postgresql:";
    private static final String H2 = "h2";
    private static final String USERNAME = "username";
    private static final String PASSWORD = "password";
    private static final String HOSTNAME = "hostname";
    private static final String PORT = "port";
    private Map<String, Object> credentials = new HashMap<>();
    DbIdentifiers.DB db = DbIdentifiers.DB.NONE;

    private DbCredentialsBuilder() {
    }

    public DbCredentials build() throws InternalError {
        switch (db) {
            case POSTGRESQL:
                return createPostgresCredentials();
            case H2:
                return createH2Credentials();
            default:
                throw new InternalError("Don't support this DB");
        }

    }

    public static DbCredentialsBuilder create() {
        return new DbCredentialsBuilder();
    }

    public DbCredentialsBuilder db(DbIdentifiers.DB db) {
        //make sure not to overwrite a db type auto determined via credentials
        if ((this.db == null || this.db == DbIdentifiers.DB.NONE)
                && (db != null && db != DbIdentifiers.DB.NONE)) {
            this.db = db;
        }
        return this;
    }

    public DbCredentialsBuilder credentials(Map<String, Object> credentials) {
        this.credentials = new HashMap<>(credentials);
        setDbType(credentials);
        return this;
    }

    private void setDbType(Map<String, Object> credentials) {
        if (credentials.containsKey(URI)) {
            String uri = (String) credentials.get(URI);
            if (StringUtils.isNotEmpty(uri)) {
                if (isDbType(uri, POSTGRES, POSTGRESQL)) {
                    db = DbIdentifiers.DB.POSTGRESQL;
                } else if (isDbType(uri, H2)) {
                    db = DbIdentifiers.DB.H2;
                }
            }
        }
    }

    private boolean isDbType(String uri, String... dbStartString) {
        return Arrays.stream(dbStartString).anyMatch(start -> uri.startsWith(JDBC + start) || uri.startsWith(start));
    }

    private DbCredentials createPostgresCredentials() throws InternalError {
        String username = (String) credentials.get(USERNAME);
        String password = (String) credentials.get(PASSWORD);
        String hostname = (String) credentials.get(HOSTNAME);
        String port = (String) credentials.get(PORT);
        String uri = (String) credentials.get(URI);
        String database = null;
        if (credentials.containsKey(DBNAME)) {
            database = (String) credentials.get(DBNAME);
        }
        if (credentials.containsKey(DATABASE)) {
            database = (String) credentials.get(DATABASE);
        }
        DbCredentialsPostgreSQL credentialsPostgreSQL = new DbCredentialsPostgreSQL(username, password, hostname, port, database, uri);
        if (StringUtils.isEmpty(credentialsPostgreSQL.getUser())) {
            throw new InternalError("No DB user specified");
        }
        if (StringUtils.isEmpty(credentialsPostgreSQL.getPassword())) {
            throw new InternalError("No DB password specified");
        }
        if (StringUtils.isEmpty(credentialsPostgreSQL.getUrl()) &&
                StringUtils.isEmpty(credentialsPostgreSQL.getHost())) {
            throw new InternalError("No DB uri and host specified");
        }
        return credentialsPostgreSQL;
    }

    private DbCredentials createH2Credentials() throws InternalError {
        String username = (String) credentials.get(USERNAME);
        String password = (String) credentials.get(PASSWORD);
        String database = null;
        if (credentials.containsKey(DBNAME)) {
            database = (String) credentials.get(DBNAME);
        }
        if (credentials.containsKey(DATABASE)) {
            database = (String) credentials.get(DATABASE);
        }
        DbCredentialsH2 credentialsH2 = new DbCredentialsH2(username, password, database);
        if (StringUtils.isEmpty(credentialsH2.getUser())) {
            throw new InternalError("No DB user specified");
        }
        return credentialsH2;
    }
}
