/*
 * *************************************************************************
 *  * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 *  *************************************************************************
 */

package com.sap.cloud.mt.subscription;

import com.sap.cloud.mt.subscription.exceptions.InternalError;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;

public class DbCredentialsH2 extends DbCredentials {
    private static final String ORG_H2_DRIVER = "org.h2.Driver";
    private final String database;
    private final String h2PackageVersion;

    public DbCredentialsH2(String user, String password, String database) throws InternalError {
        super(user, password, "", "", ORG_H2_DRIVER, "");
        this.h2PackageVersion = getH2PackageVersion();
        this.database = database;
    }

    public DbCredentialsH2(DbCredentialsH2 dbCredentials) {
        super(dbCredentials);
        this.h2PackageVersion = getH2PackageVersion();
        this.database = dbCredentials.database;
    }

    @Override
    public String getDatabaseId() {
        return database;
    }

    @Override
    protected String buildUrl() {
        /**
         * Liquibase has a severe problem with H2 databases. Its handling of its own database table DATABASECHANGELOG is inconsistent
         * concerning upper and lower case characters on this DB. It seems to check for its existence with another notation then used for its creation.
         * This results in errors "DATABASECHANGELOG already exists" when it is tried to create it wrongly a second time. See for example
         * https://github.com/liquibase/liquibase-cache/issues/1 for a discussion of this problem. One workaround is to configure H2 to ignore
         * upper and lower case by using parameter COLLATION=DEFAULT_EN STRENGTH PRIMARY.
         */
        if (h2PackageVersion.startsWith("1.")) {
            return "jdbc:h2:mem:" + database + ";DB_CLOSE_DELAY=-1;DATABASE_TO_LOWER=TRUE;CASE_INSENSITIVE_IDENTIFIERS=TRUE;";
        } else {
            return "jdbc:h2:mem:" + database + ";DB_CLOSE_DELAY=-1;COLLATION=DEFAULT_EN STRENGTH PRIMARY;";
        }

    }

    @Override
    public DbIdentifiers.DB getDB() {
        return DbIdentifiers.DB.H2;
    }

    @Override
    public DbCredentials createCopy() {
        return new DbCredentialsH2(this);
    }

    @Override
    protected List<HostAndPort> getHostsFromUri(URI uri) {
        return new ArrayList<>();
    }

    @Override
    protected UserAndPassword getUserFromUri(URI uri) {
        return new UserAndPassword();
    }

    static String getH2PackageVersion() {
        try {
            Class<?> c = Class.forName(ORG_H2_DRIVER);//NOSONAR
            return c.getPackage().getImplementationVersion();
        } catch (ClassNotFoundException e) {
            throw new RuntimeException("Class org.h2.Driver not available");//NOSONAR
        }
    }
}
