/*
 * *************************************************************************
 *  * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 *  *************************************************************************
 */

package com.sap.cloud.mt.subscription;

import com.sap.cloud.mt.subscription.exceptions.InternalError;
import org.apache.commons.lang3.StringUtils;

import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class DbCredentialsPostgreSQL extends DbCredentials {
    private static final String URL_TEMPLATE = "jdbc:postgresql://{host}:{port}/{database}";
    private static final String URL_TEMPLATE_WITHOUT_PORT = "jdbc:postgresql://{host}/{database}";
    private static final String ORG_POSTGRESQL_DRIVER = "org.postgresql.Driver";
    private final String database;

    public DbCredentialsPostgreSQL(String user, String password, String host, String port, String database, String uriStr) throws InternalError {
        super(user, password, host, port, ORG_POSTGRESQL_DRIVER, uriStr != null ? uriStr.replace("postgres:", "postgresql:") : "");
        this.database = database;
    }

    public DbCredentialsPostgreSQL(DbCredentialsPostgreSQL dbCredentials) {
        super(dbCredentials);
        this.database = dbCredentials.database;
    }

    @Override
    public String getDatabaseId() {
        if (StringUtils.isNotEmpty(database)) {
            return database;
        } else {
            if (uri == null) {
                return "";
            }
            String path = uri.getPath();
            if (StringUtils.isNotEmpty(path)) {
                return path.substring(1);
            } else {
                return "";
            }
        }
    }

    @Override
    protected String buildUrl() {
        if (StringUtils.isNotEmpty(getPort())) {
            return URL_TEMPLATE.replace("{host}", getHost()).replace("{port}", getPort()).replace("{database}", getDatabaseId());
        } else {
            return URL_TEMPLATE_WITHOUT_PORT.replace("{host}", getHost()).replace("{database}", getDatabaseId());
        }
    }

    @Override
    public DbIdentifiers.DB getDB() {
        return DbIdentifiers.DB.POSTGRESQL;
    }

    @Override
    public DbCredentials createCopy() {
        return new DbCredentialsPostgreSQL(this);
    }

    @Override
    protected List<HostAndPort> getHostsFromUri(URI uri) {
        List<HostAndPort> hostAndPorts = new ArrayList<>();
        String authority = uri.getAuthority();
        if (StringUtils.isEmpty(authority)) {
            return hostAndPorts;
        }
        String hostsStr = optionalADividerB_getB("@", authority);
        String[] hosts = hostsStr.split(",", 0);
        Arrays.stream(hosts).forEach(h -> {
            HostAndPort hostAndPort = new HostAndPort();
            hostAndPort.host = aDividerOptionalB_getA(":", h);
            hostAndPort.port = aDividerOptionalB_getB(":", h);
            hostAndPorts.add(hostAndPort);
        });
        return hostAndPorts;
    }

    @Override
    protected UserAndPassword getUserFromUri(URI uri) {
        UserAndPassword userAndPassword = new UserAndPassword();
        userAndPassword.user = "";
        userAndPassword.password = "";
        String authority = uri.getAuthority();
        if (StringUtils.isEmpty(authority)) {
            return userAndPassword;
        }
        String userStr = optionalADividerB_getA("@", authority);
        userAndPassword.user = aDividerOptionalB_getA(":", userStr);
        userAndPassword.password = aDividerOptionalB_getB(":", userStr);
        return userAndPassword;
    }

    private String optionalADividerB_getA(String divider, String str) {
        if (!str.contains(divider)) {
            return "";
        }
        String[] components = str.split(divider, 2);
        return components[0];
    }

    private String optionalADividerB_getB(String divider, String str) {
        if (!str.contains(divider)) {
            return str;
        }
        String[] components = str.split(divider, 2);
        return components[1];
    }

    private String aDividerOptionalB_getA(String divider, String str) {
        if (!str.contains(divider)) {
            return str;
        }
        String[] components = str.split(divider, 2);
        return components[0];
    }

    private String aDividerOptionalB_getB(String divider, String str) {
        if (!str.contains(divider)) {
            return "";
        }
        String[] components = str.split(divider, 2);
        return components[1];
    }
}
