/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/
package com.sap.cloud.mt.subscription;

import com.sap.cloud.mt.subscription.exceptions.InternalError;
import com.sap.cloud.mt.subscription.exceptions.UnknownTenant;

import java.util.List;
import java.util.Map;
import java.util.Set;

public interface InstanceLifecycleManager {
    public static final String DATABASE_ID = "database_id";

    public void createNewInstance(String tenantId, ProvisioningParameters instanceParameters,
                                  BindingParameters bindingParameters) throws InternalError;

    public void deleteInstance(String tenantId) throws InternalError;

    public DataSourceInfo getDataSourceInfo(String tenantId, boolean forceCacheUpdate) throws InternalError, UnknownTenant;

    public ContainerStatus getContainerStatus(String tenantId) throws InternalError;

    /**
     * Checks if the tenant has credentials for the schema.
     *
     * @param tenantId identifier of tenant that is checked
     * @param forceCacheUpdate If set, the current tenant is re-determined.
     * @return true if credentials are available
     * @throws InternalError is thrown if a problem occurs, if for example service manager cannot be accessed
     */
    public boolean hasCredentials(String tenantId, boolean forceCacheUpdate) throws InternalError;

    /**
     * Returns all tenants that have a DB container/schema. In contrast to {@link #getAllTenants(boolean)}
     * this method provides additional tenant properties like database_id if available.
     *
     * @param forceCacheUpdate If set, the current list of tenants is re-determined.
     * @return A map with all {@link TenantMetadata}. Tenant id is used as key.
     * @throws InternalError
     */
    public Map<String, TenantMetadata> getAllTenantInfos(boolean forceCacheUpdate) throws InternalError;

    /**
     * Returns all tenants that have a DB container/schema
     *
     * @param forceCacheUpdate If set, the current list of tenants is re-determined.
     * @return A set with all tenant ids
     * @throws InternalError
     */
    default Set<String> getAllTenants(boolean forceCacheUpdate) throws InternalError {
        return getAllTenantInfos(forceCacheUpdate).keySet();
    }

    /**
     * Checks if a schema has been created for a tenant
     *
     * @param tenantId identifier of tenant that is checked
     * @throws UnknownTenant is thrown if tenant doesn't exist
     * @throws InternalError is thrown if a problem occurs, if for example service manager or DB cannot be accessed
     */
    public void checkThatTenantExists(String tenantId) throws UnknownTenant, InternalError;

    /**
     * State of an HDI container.
     * The service manager provides the results of the last operation on instance level. If for example a patch operation
     * is triggered and fails, the last operation will show a task update with status failed. This means the last operation
     * state does not indicate the state of the container itself. Even if an update or delete operation failed, the container
     * is still working. Only if the operation is of type create the container status can be derived from it. Therefore,
     * only a limited sets of states are defined here.
     */
    public static enum ContainerStatus {
        OK("ok"), CREATION_IN_PROGRESS("creation in progress"), DOES_NOT_EXIST("doesn't exist"), CREATION_ERROR("creation error");
        private String status;

        ContainerStatus(String status) {
            this.status = status;
        }

        @Override
        public String toString() {
            return status;
        }
    }

    /**
     * Creates on the database used by the given datasource a hdi container/schema reserved of the library itself.
     * If the parameter is null or the database cannot be determined, the missing containers are created on all databases.
     * When used with Hana a service instance is created for tenant MT_LIB_TENANT-"database id"
     * No error is thrown if container creation failed !
     *
     * @param dataSourceInfo Credentials of dataSource for which lib container is created
     * @return Info for lib container
     * @throws InternalError is thrown when a communication problems arises
     */
    public List<DataSourceInfo> createAndGetLibContainers(DataSourceInfo dataSourceInfo) throws InternalError;

    /**
     * @return the credentials of mt-libs's own containers/schemas. In case of HANA, the list is based
     * on the current state of the instance manager client libs cache.
     * @throws InternalError is thrown when a communication problems arises
     */
    public List<DataSourceInfo> getLibContainers() throws InternalError;

    /**
     * The mt-lib creates a container/schema in each provided database for itself. This function returns
     * the fake tenantId which is used for it.
     *
     * @param databaseId
     * @return tenantId for mt-lib schema/container
     */
    public default String getMtLibContainerName(String databaseId) {
        return FilterTenants.MT_LIB_TENANT_PREFIX + databaseId;
    }

    /**
     * Create InstanceCreateOptions that contain the specified database id
     *
     * @param databaseId
     * @return new instance create options with set database id
     */
    public default ProvisioningParameters createProvisioningParameters(String databaseId) {
        var provisioningParameters = new ProvisioningParameters();
        provisioningParameters.put(DATABASE_ID, databaseId);
        return provisioningParameters;
    }

    public boolean hasDbIdentifiers();

    public void insertDbIdentifiers(DbIdentifiers dbIdentifiers);

    default DbIdentifiers.DB getDbType() {
        return DbIdentifiers.DB.HANA;
    }

    default boolean knowsDbCredentials() {
        return true;
    }
}
