/*******************************************************************************
 *   © 2019-2024 SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.mt.subscription;

import com.sap.cloud.mt.subscription.ServiceOperation.Status;
import com.sap.cloud.mt.subscription.ServiceOperation.Type;
import org.apache.commons.lang3.StringUtils;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Service instance parameters
 */
public class ServiceInstance extends HashMap<String, Object> {
    public static final String ID = "id";
    public static final String NAME = "name";
    private List<ServiceBinding> bindings = new ArrayList<>();
    private ServiceBinding newestBinding = null;
    private final ServiceOperation lastOperation;

    public ServiceInstance(Map<String, ?> m) {
        super(m);
        lastOperation = new ServiceOperation(m != null ? m : new HashMap<String, Object>());

    }

    public String getId() {
        return containsKey(ID) ? (String) get(ID) : "";
    }

    public boolean isReady() {
        return containsKey("ready") && (boolean) get("ready");
    }

    public String getName() {
        return containsKey(NAME) ? (String) get(NAME) : "";
    }

    public String getServicePlanId() {
        return (String) get("service_plan_id");
    }

    public List<String> getTenants() {
        var labels = (Map<String, Object>) get("labels");
        if (labels == null) {
            return new ArrayList<>();
        }
        var tenants = (List<String>) labels.get("tenant_id");
        return tenants != null ? tenants.stream().filter(t -> !StringUtils.isBlank(t)).toList() : new ArrayList<>();
    }

    public boolean hasTenant() {
        return !getTenants().isEmpty();
    }

    public Instant getCreatedAt() {
        return Instant.parse((String) get("created_at"));
    }

    public Instant getUpdatedAt() {
        return Instant.parse((String) get("updated_at"));
    }

    public String getUsable() {
        return (String) get("usable");
    }

    public ServiceOperation getLastOperation() {
        return lastOperation;
    }

    public List<ServiceBinding> getBindings() {
        return bindings;
    }

    public void setBindings(List<ServiceBinding> bindings) {
        this.bindings = bindings != null ? bindings : new ArrayList<>();
        final AtomicReference<ServiceBinding> newestBindingRef = new AtomicReference<>(null);
        this.bindings.stream()
                .filter(b -> b.getServiceInstanceId().equals(getId()))
                .filter(b -> bindingTenantIsContainedInInstanceTenant(getTenants(), b.getTenants()))
                .filter(ServiceBinding::isUsable).forEach(b -> {
                    if (newestBindingRef.get() == null ||
                            b.getCreatedAt().isAfter(newestBindingRef.get().getCreatedAt())) {
                        newestBindingRef.set(b);
                    }
                });
        if (newestBindingRef.get() == null) {
            this.newestBinding = null;
        } else {
            this.newestBinding = newestBindingRef.get();
        }
    }

    public Optional<ServiceBinding> getBinding() {
        return newestBinding != null ? Optional.of(newestBinding) : Optional.empty();
    }

    public boolean isUsable() {
        var status = lastOperation.getStatus();
        return isReady() &&
                !(lastOperation.getType() == Type.CREATE && (status == Status.FAILED || status == Status.IN_PROGRESS));
    }

    private boolean bindingTenantIsContainedInInstanceTenant(List<String> instanceTenants, List<String> bindingTenants) {
        return instanceTenants.stream().anyMatch(bindingTenants::contains);
    }
}
