/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/
package com.sap.cloud.mt.subscription;

import com.sap.cloud.mt.subscription.exceptions.AuthorityError;
import com.sap.cloud.mt.subscription.exceptions.InternalError;
import com.sap.cloud.mt.subscription.exceptions.NotFound;
import com.sap.cloud.mt.subscription.exceptions.NotSupported;
import com.sap.cloud.mt.subscription.exceptions.ParameterError;
import com.sap.cloud.mt.subscription.json.ApplicationDependency;
import com.sap.cloud.mt.subscription.json.DeletePayload;
import com.sap.cloud.mt.subscription.json.SubscriptionPayload;

import java.util.List;


/**
 * Interface Subscriber provides methods to subscribe new tenants, start hdi deployments and unsubscribe tenants.
 * It works together with the Hana instance manager, that provides a rest API to create and delete hdi containers for tenants, and the dynamic
 * HDI deployment app, a node.js application that contains the database artifacts definitions and deploys them into an hdi container if called via its
 * rest API.
 */
public interface Subscriber {

    /**
     * Unsubscribes the specified tenant synchronously
     *
     * @param tenantId      the tenant identifier
     * @param deletePayload Information provided by CIS about the subscribed application
     * @throws InternalError  is thrown if instance manager reports an error
     * @throws AuthorityError is thrown in case of missing authority
     * @throws ParameterError is thrown if needed configuration parameters aren't set
     */
    void unsubscribe(String tenantId, DeletePayload deletePayload) throws InternalError, ParameterError, AuthorityError;

    /**
     * Returns the list of microservices the program depends on
     *
     * @return list of dependent microservices
     */
    List<ApplicationDependency> getApplicationDependencies() throws AuthorityError;

    /**
     * Subscribes the specified tenant synchronously.
     *
     * @param tenantId            is the tenant identifier
     * @param subscriptionPayload is provided by CIS and describes the subscribing application
     * @return The URL of the application UI
     * @throws InternalError  is thrown if instance manager or hdi deployment app report an error
     * @throws AuthorityError is thrown in case of missing authority
     * @throws ParameterError is thrown if needed configuration parameters aren't set
     */
    String subscribe(String tenantId, SubscriptionPayload subscriptionPayload) throws InternalError, ParameterError, AuthorityError;

    /**
     * Gets the URL of the saas application
     *
     * @param subscriptionPayload is provided by CIS and describes the subscribing application
     * @return The URL of the application UI
     * @throws InternalError  is thrown if instance manager or hdi deployment app report an error
     * @throws AuthorityError is thrown in case of missing authority
     * @throws ParameterError is thrown if needed configuration parameters aren't set
     */
    String getSubscribeUrl(SubscriptionPayload subscriptionPayload) throws InternalError, ParameterError, AuthorityError;

    /**
     * Starts hdi deployment for the specified already subscribed tenants.
     *
     * @param tenants is the list of tenant identifiers, if it contains only one entry "all" all tenants are updated
     * @throws InternalError  is thrown if the hdi deployment app reports an error
     * @throws AuthorityError is thrown in case of missing authority
     * @throws ParameterError is thrown if needed configuration parameters aren't set
     */
    void setupDbTables(List<String> tenants) throws InternalError, NotSupported, ParameterError, AuthorityError;

    /**
     * Starts hdi deployment for the specified already subscribed tenants asynchronously.
     *
     * @param tenants is the list of tenant identifiers, if it contains only one entry "all" all tenants are updated
     * @return JobId if sidecar is used, otherwise empty
     * @throws InternalError  is thrown if the hdi deployment app or sidecar reports an error
     * @throws AuthorityError is thrown in case of missing authority
     * @throws ParameterError is thrown if needed configuration parameters aren't set
     */
    String setupDbTablesAsync(List<String> tenants) throws InternalError, ParameterError, AuthorityError;

    /**
     * Returns the status of an asynchronous upgrade operation. Only supported for sidecar.
     *
     * @param jobId returned from sidecar
     * @return Returns a sidecar json :{
     * "error": null,
     * "status": "RUNNING",
     * "result": null
     * }
     * @throws InternalError  is thrown if sidecar cannot be accessed.
     * @throws AuthorityError is thrown in case of missing authority
     * @throws ParameterError is thrown if needed configuration parameters aren't set
     * @throws NotSupported   is thrown if operation is not supported by subscriber implementation
     * @throws NotFound       is thrown if is jobId not found
     */
    String updateStatus(String jobId) throws NotSupported, InternalError, ParameterError, NotFound, AuthorityError;

    /**
     * Calls the saas registry.
     *
     * @param ok              Operation worked.
     * @param message         Message returned to customers.
     * @param applicationUrl  Must be filled in case of subscription. The URL under which the UI of the subscribed application can be reached.
     * @param saasRegistryUrl The callback URL provided by CIS.
     * @throws InternalError Thrown in case the saas registry couldn't be called.
     */
    void callSaasRegistry(boolean ok, String message, String applicationUrl, String saasRegistryUrl) throws InternalError;

    /**
     * @throws AuthorityError thrown if authority is missing
     */
    void checkAuthority(SecurityChecker.Authority authority) throws AuthorityError;
}
