/*
 * *************************************************************************
 *  * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 *  *************************************************************************
 */

package com.sap.cloud.mt.subscription;

import com.sap.cloud.mt.subscription.HanaEncryptionTool.DbEncryptionMode;
import com.sap.cloud.mt.subscription.exceptions.InternalError;
import com.sap.cloud.mt.subscription.exits.Exits;
import com.sap.cloud.mt.tools.api.ResilienceConfig;
import org.apache.commons.lang3.StringUtils;

public class SubscriberBuilder {
    private String baseUiUrl;
    private String urlSeparator;
    private Exits exits;
    private SecurityChecker securityChecker;
    private SaasRegistry saasRegistry;
    private InstanceLifecycleManager instanceLifecycleManager;
    private HdiContainerManager hdiContainerManager;
    private LiquibaseParameters liquibaseParameters;
    private ProvisioningService provisioningService;
    private boolean withoutAuthorityCheck = false;
    private DbEncryptionMode hanaEncryptionMode;
    private ResilienceConfig resilienceConfig = ResilienceConfig.NONE;

    private SubscriberBuilder() {
    }

    public static SubscriberBuilder create() {
        return new SubscriberBuilder();
    }

    public Subscriber build() throws InternalError {
        if (provisioningService != null) {
            return SubscriberStreamlinedMtx.Builder.create()
                    .provisioningService(provisioningService)
                    .baseUiUrl(baseUiUrl)
                    .urlSeparator(urlSeparator)
                    .exits(exits)
                    .securityChecker(securityChecker)
                    .saasRegistry(saasRegistry)
                    .instanceLifecycleManager(instanceLifecycleManager)
                    .withoutAuthorityCheck(withoutAuthorityCheck)
                    .hanaEncryptionMode(hanaEncryptionMode)
                    .build();
        } else if (hdiContainerManager != null) {
            return new SubscriberImpl(instanceLifecycleManager,
                    hdiContainerManager, baseUiUrl, urlSeparator,
                    exits, securityChecker, saasRegistry, withoutAuthorityCheck, hanaEncryptionMode, resilienceConfig);
        } else if (liquibaseParameters != null
                && (StringUtils.isNotBlank(liquibaseParameters.getFileName()) || liquibaseParameters.getResourceAccessor() != null)) {
            DbDeployer dbDeployer = new DbDeployerLiquibase(liquibaseParameters);
            return new SubscriberImpl(instanceLifecycleManager, dbDeployer, baseUiUrl, urlSeparator, exits,
                    securityChecker, saasRegistry, withoutAuthorityCheck, hanaEncryptionMode, resilienceConfig);
        } else {
            throw new InternalError("Could not create subscriber");
        }

    }

    public SubscriberBuilder baseUiUrl(String baseUiUrl) {
        this.baseUiUrl = baseUiUrl;
        return this;
    }

    public SubscriberBuilder urlSeparator(String urlSeparator) {
        this.urlSeparator = urlSeparator;
        return this;
    }

    public SubscriberBuilder exits(Exits exits) {
        this.exits = exits;
        return this;
    }

    public SubscriberBuilder securityChecker(SecurityChecker securityChecker) {
        this.securityChecker = securityChecker;
        return this;
    }

    public SubscriberBuilder saasRegistry(SaasRegistry saasRegistry) {
        this.saasRegistry = saasRegistry;
        return this;
    }

    public SubscriberBuilder instanceLifecycleManager(InstanceLifecycleManager instanceLifecycleManager) {
        this.instanceLifecycleManager = instanceLifecycleManager;
        return this;
    }

    public SubscriberBuilder hdiContainerManager(HdiContainerManager hdiContainerManager) {
        this.hdiContainerManager = hdiContainerManager;
        return this;
    }

    public SubscriberBuilder liquibaseParameters(LiquibaseParameters liquibaseParameters) {
        this.liquibaseParameters = liquibaseParameters;
        return this;
    }

    public SubscriberBuilder provisioningService(ProvisioningService provisioningService) {
        this.provisioningService = provisioningService;
        return this;
    }

    public SubscriberBuilder withoutAuthorityCheck(boolean withoutAuthorityCheck) {
        this.withoutAuthorityCheck = withoutAuthorityCheck;
        return this;
    }

    public SubscriberBuilder hanaEncryptionMode(DbEncryptionMode hanaEncryptionMode) {
        this.hanaEncryptionMode = hanaEncryptionMode;
        return this;
    }

    public SubscriberBuilder resilienceConfig(ResilienceConfig resilienceConfig) {
        this.resilienceConfig = resilienceConfig;
        return this;
    }
}
